import logging
import pdb
from typing import Any, Callable, List, Optional, Type, Union

import torch
import torch.nn as nn
from torch import Tensor
from torch.hub import load_state_dict_from_url
from torchmeta.modules import (MetaBatchNorm2d, MetaConv2d, MetaLinear,
                               MetaModule, MetaSequential)

logger = logging.getLogger()

# Wrappers of Pytorch official ResNet class
# See https://github.com/pytorch/vision/blob/master/torchvision/models/resnet.py


__all__ = ['ResNet', 'resnet18', 'resnet34', 'resnet50', 'resnet101',
           'resnet152', 'resnext50_32x4d', 'resnext101_32x8d',
           'wide_resnet50_2', 'wide_resnet101_2']


model_urls = {
    'resnet18': 'https://download.pytorch.org/models/resnet18-f37072fd.pth',
    'resnet34': 'https://download.pytorch.org/models/resnet34-b627a593.pth',
    'resnet50': 'https://download.pytorch.org/models/resnet50-0676ba61.pth',
    'resnet101': 'https://download.pytorch.org/models/resnet101-63fe2227.pth',
    'resnet152': 'https://download.pytorch.org/models/resnet152-394f9c45.pth',
    'resnext50_32x4d': 'https://download.pytorch.org/models/resnext50_32x4d-7cdf4587.pth',
    'resnext101_32x8d': 'https://download.pytorch.org/models/resnext101_32x8d-8ba56ff5.pth',
    'wide_resnet50_2': 'https://download.pytorch.org/models/wide_resnet50_2-95faca4d.pth',
    'wide_resnet101_2': 'https://download.pytorch.org/models/wide_resnet101_2-32ee1156.pth',
}


def conv3x3(in_planes: int, out_planes: int, stride: int = 1, groups: int = 1, dilation: int = 1) -> MetaConv2d:
    """3x3 convolution with padding"""
    return MetaConv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                      padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes: int, out_planes: int, stride: int = 1) -> MetaConv2d:
    """1x1 convolution"""
    return MetaConv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(MetaModule):
    expansion: int = 1

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[MetaModule] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., MetaModule]] = None
    ) -> None:
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = MetaBatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor, params=None) -> Tensor:
        identity = x

        out = self.conv1(x, params=self.get_subdict(params, "conv1"))
        out = self.bn1(out, params=self.get_subdict(params, "bn1"))
        out = self.relu(out)

        out = self.conv2(out, params=self.get_subdict(params, "conv2"))
        out = self.bn2(out, params=self.get_subdict(params, "bn2"))

        if self.downsample is not None:
            identity = self.downsample(x, params=self.get_subdict(params, "downsample"))

        out += identity
        out = self.relu(out)

        return out


class Bottleneck(MetaModule):
    # Bottleneck in torchvision places the stride for downsampling at 3x3 convolution(self.conv2)
    # while original implementation places the stride at the first 1x1 convolution(self.conv1)
    # according to "Deep residual learning for image recognition"https://arxiv.org/abs/1512.03385.
    # This variant is also known as ResNet V1.5 and improves accuracy according to
    # https://ngc.nvidia.com/catalog/model-scripts/nvidia:resnet_50_v1_5_for_pytorch.

    expansion: int = 4

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[MetaModule] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., MetaModule]] = None
    ) -> None:
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = MetaBatchNorm2d
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor, params=None) -> Tensor:
        identity = x

        out = self.conv1(x, params=self.get_subdict(params, "conv1"))
        out = self.bn1(out, params=self.get_subdict(params, "bn1"))
        out = self.relu(out)

        out = self.conv2(out, params=self.get_subdict(params, "conv2"))
        out = self.bn2(out, params=self.get_subdict(params, "bn2"))
        out = self.relu(out)

        out = self.conv3(out, params=self.get_subdict(params, "conv3"))
        out = self.bn3(out, params=self.get_subdict(params, "bn3"))

        if self.downsample is not None:
            identity = self.downsample(x, params=self.get_subdict(params, "downsample"))

        out += identity
        out = self.relu(out)

        return out


class ResNet(MetaModule):

    def __init__(
        self,
        block: Type[Union[BasicBlock, Bottleneck]],
        layers: List[int],
        num_classes: int = 1000,
        zero_init_residual: bool = False,
        groups: int = 1,
        width_per_group: int = 64,
        replace_stride_with_dilation: Optional[List[bool]] = None,
        norm_layer: Optional[Callable[..., MetaModule]] = None
    ) -> None:
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = MetaBatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = MetaConv2d(3, self.inplanes, kernel_size=7, stride=2, padding=3,
                                bias=False)
        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = MetaLinear(512 * block.expansion, num_classes)

        for m in self.modules():
            if isinstance(m, MetaConv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (MetaBatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)  # type: ignore[arg-type]
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)  # type: ignore[arg-type]

    def _make_layer(self, block: Type[Union[BasicBlock, Bottleneck]], planes: int, blocks: int,
                    stride: int = 1, dilate: bool = False) -> MetaSequential:
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = MetaSequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return MetaSequential(*layers)

    def _forward_impl(self, x: Tensor, params=None) -> Tensor:
        # See note [TorchScript super()]
        x = self.conv1(x, params=self.get_subdict(params, "conv1"))
        x = self.bn1(x, params=self.get_subdict(params, "bn1"))
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x, params=self.get_subdict(params, "layer1"))
        x = self.layer2(x, params=self.get_subdict(params, "layer2"))
        x = self.layer3(x, params=self.get_subdict(params, "layer3"))
        x = self.layer4(x, params=self.get_subdict(params, "layer4"))

        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x, params=self.get_subdict(params, "fc"))

        return x

    def forward(self, x: Tensor, params=None) -> Tensor:
        return self._forward_impl(x, params)


class BaseResNet(ResNet):
    def __init__(self, num_classes, pretrained=False, finetune=False, modified=False, progress=True, **kwargs):
        super(BaseResNet, self).__init__(block=self.block_class, layers=self.layers, **kwargs)
        if pretrained:
            if isinstance(pretrained, bool):
                logger.info(f"## Load pretrained weight from {model_urls[self.arch]}")
                state_dict = load_state_dict_from_url(model_urls[self.arch], progress=progress)
                self.load_state_dict(state_dict, strict=False)
            elif isinstance(pretrained, str) and not modified:
                logger.info(f"## Load pretrained weight from {pretrained}")
                state_dict = torch.load(pretrained, map_location="cpu")
                self.load_state_dict(state_dict, strict=False)
        self.num_classes = num_classes
        if finetune:
            logger.info(f"## Init. FC layer for {num_classes} classes")
            if self.block_class is BasicBlock:
                self.fc = MetaLinear(512, num_classes)
            else:
                self.fc = MetaLinear(2048, num_classes)
            self.fc.weight.data.normal_(mean=0.0, std=0.01)
            self.fc.bias.data.zero_()
            if modified:
                logger.info(f"## Load pretrained weight from {pretrained}")
                state_dict = torch.load(pretrained, map_location="cpu")
                if "model_classifier" in state_dict:
                    self.load_state_dict(state_dict["model_classifier"], strict=True)
                else:
                    self.load_state_dict(state_dict, strict=True)


class BaseResNetFeat(BaseResNet):
    def __init__(self, num_classes, pretrained=False, finetune=False, modified=False, progress=True, **kwargs):
        super().__init__(num_classes, pretrained, finetune, modified, progress, **kwargs)

    def _forward_impl(self, x: Tensor, params=None) -> Tensor:
        # See note [TorchScript super()]
        x = self.conv1(x, params=self.get_subdict(params, "conv1"))
        x = self.bn1(x, params=self.get_subdict(params, "bn1"))
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x, params=self.get_subdict(params, "layer1"))
        x = self.layer2(x, params=self.get_subdict(params, "layer2"))
        x = self.layer3(x, params=self.get_subdict(params, "layer3"))
        x = self.layer4(x, params=self.get_subdict(params, "layer4"))

        feat = self.avgpool(x)
        pred = self.fc(torch.flatten(feat, 1), params=self.get_subdict(params, "fc"))

        return pred, feat


class ResNet18(BaseResNet):
    def __init__(self, *args, **kwargs):
        self.arch = "resnet18"
        self.block_class = BasicBlock
        self.layers = [2, 2, 2, 2]
        super(ResNet18, self).__init__(*args, **kwargs)


class ResNet18Feat(BaseResNetFeat):
    def __init__(self, *args, **kwargs):
        self.arch = "resnet18"
        self.block_class = BasicBlock
        self.layers = [2, 2, 2, 2]
        super(ResNet18Feat, self).__init__(*args, **kwargs)


class ResNet50(BaseResNet):
    def __init__(self, *args, **kwargs):
        self.arch = "resnet50"
        self.block_class = Bottleneck
        self.layers = [3, 4, 6, 3]
        super(ResNet50, self).__init__(*args, **kwargs)


class ResNet50Feat(BaseResNetFeat):
    def __init__(self, *args, **kwargs):
        self.arch = "resnet50"
        self.block_class = Bottleneck
        self.layers = [3, 4, 6, 3]
        super(ResNet50Feat, self).__init__(*args, **kwargs)
