# Copyright 2019, The TensorFlow Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Training a language model (recurrent neural network) with DP-SGD optimizer.

This tutorial uses a corpus of text from TensorFlow datasets unless a
FLAGS.data_dir is specified with the path to a directory containing two files
train.txt and test.txt corresponding to a training and test corpus.

Even though we haven't done any hyperparameter tuning, and the analytical
epsilon upper bound can't offer any strong guarantees, the benefits of training
with differential privacy can be clearly seen by examining the trained model.
In particular, such inspection can confirm that the set of training-data
examples that the model fails to learn (i.e., has high perplexity for) comprises
outliers and rare sentences outside the distribution to be learned (see examples
and a discussion in this blog post). This can be further confirmed by
testing the differentially-private model's propensity for memorization, e.g.,
using the exposure metric of https://arxiv.org/abs/1802.08232.

This example is decribed in more details in this post: https://goo.gl/UKr7vH
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from absl import app
from absl import flags
import time
import numpy as np
import tensorflow as tf
print(tf.__version__)
print("Num GPUs Available: ", len(tf.config.experimental.list_physical_devices('GPU')))
from keras.preprocessing import sequence

from tensorflow_privacy.privacy.optimizers import dp_optimizer
import pickle

BATCH_SIZE=256
flags.DEFINE_boolean(
    'dpsgd', True, 'If True, train with DP-SGD. If False, '
    'train with vanilla SGD.')
flags.DEFINE_float('learning_rate', 0.001, 'Learning rate for training')
flags.DEFINE_float('noise_multiplier', 0.6,
                   'Ratio of the standard deviation to the clipping norm')
flags.DEFINE_float('l2_norm_clip', 1.0, 'Clipping norm')
flags.DEFINE_integer('batch_size', BATCH_SIZE, 'Batch size')
flags.DEFINE_integer('epochs', 30, 'Number of epochs')
flags.DEFINE_integer(
    'microbatches', BATCH_SIZE, 'Number of microbatches '
    '(must evenly divide batch_size)') # time proportional to this
flags.DEFINE_string('model_dir', None, 'Model directory')

FLAGS = flags.FLAGS

SEQ_LEN = 150
maxfeature=8185

tf.random.set_seed(1)
from tensorflow.keras.layers import Dense, Embedding,Bidirectional,LSTM


def rnn_model_fn(features, labels, mode):  # pylint: disable=unused-argument
  """Model function for a RNN."""

  # Define RNN architecture using tf.keras.layers.
  x = features
  x=tf.keras.layers.Embedding(maxfeature,64,input_length=SEQ_LEN)(x)
  #y = tf.keras.layers.Bidirectional(tf.compat.v1.keras.layers.CuDNNLSTM(64))(x)
  y = tf.keras.layers.Bidirectional(LSTM(64))(x)
  y = tf.keras.layers.Dense(64, activation='relu')(y)
  logits = tf.keras.layers.Dense(2)(y)

  # Calculate loss as a vector (to support microbatches in DP-SGD).
  vector_loss = tf.nn.sparse_softmax_cross_entropy_with_logits(
      labels=labels,logits=logits)
  # Define mean of loss across minibatch (for reporting through tf.Estimator).
  scalar_loss = tf.reduce_mean(vector_loss)

  # Configure the training op (for TRAIN mode).
  if mode == tf.estimator.ModeKeys.TRAIN:
    if FLAGS.dpsgd:
      optimizer = dp_optimizer.DPAdamGaussianOptimizer(
          l2_norm_clip=FLAGS.l2_norm_clip,
          noise_multiplier=FLAGS.noise_multiplier,
          num_microbatches=FLAGS.microbatches,
          learning_rate=FLAGS.learning_rate,
          unroll_microbatches=True,
          beta1=0.9,
          beta2=0.999,
          epsilon=1e-8)
      opt_loss = vector_loss
    else:
      optimizer = tf.compat.v1.train.AdamOptimizer(
          learning_rate=FLAGS.learning_rate)
      opt_loss = scalar_loss
    global_step = tf.compat.v1.train.get_global_step()
    train_op = optimizer.minimize(loss=opt_loss, global_step=global_step)
    return tf.estimator.EstimatorSpec(mode=mode,
                                      loss=scalar_loss,
                                      train_op=train_op)

  # Add evaluation metrics (for EVAL mode).
  elif mode == tf.estimator.ModeKeys.EVAL:
    eval_metric_ops = {
        'accuracy':
            tf.compat.v1.metrics.accuracy(
                labels=labels,
                predictions=tf.argmax(input=logits, axis=1))
    }
    return tf.estimator.EstimatorSpec(mode=mode,
                                      loss=scalar_loss,
                                      eval_metric_ops=eval_metric_ops)


NUM_SAMPLES=25000

def load_data():
  """Load training and validation data."""
  (train_data,train_labels), (test_data,test_labels) = tf.keras.datasets.imdb.load_data(num_words=maxfeature)
  print(len(train_data))
  train_data = sequence.pad_sequences(train_data[:NUM_SAMPLES], maxlen=SEQ_LEN)#.astype('int32')
  test_data = sequence.pad_sequences(test_data, maxlen=SEQ_LEN)#.astype('int32')
  train_labels = np.array(train_labels[:NUM_SAMPLES], dtype=np.int32).flatten()
  test_labels = np.array(test_labels, dtype=np.int32).flatten()
  # https://www.tensorflow.org/tutorials/text/word_embeddings
  #embedding_layer =tf.keras.layers.Embedding(10000, 100)
  #result = embedding_layer(train_data)

  assert train_labels.ndim == 1
  assert test_labels.ndim == 1

  return train_data, train_labels, test_data, test_labels

def main(unused_argv):
    
  print('***********Batch size: ',FLAGS.batch_size)
  tf.compat.v1.logging.set_verbosity(3)

  # Load training and test data.
  train_data, train_labels, test_data, test_labels = load_data()

  # Instantiate the tf.Estimator.
  #conf = tf.estimator.RunConfig(save_summary_steps=1000)
  lm_classifier = tf.estimator.Estimator(model_fn=rnn_model_fn,
                                         model_dir=FLAGS.model_dir)
#                                         ,config=conf)

  # Create tf.Estimator input functions for the training and test data.
  eval_input_fn = tf.compat.v1.estimator.inputs.numpy_input_fn(
      x=test_data,
      y=test_labels,
      num_epochs=1,
      shuffle=False)
  train_input_fn = tf.compat.v1.estimator.inputs.numpy_input_fn(
      x=train_data,
      y=train_labels,
      batch_size=FLAGS.batch_size,
      num_epochs=FLAGS.epochs,
      shuffle=True)

  # Training loop.
  steps_per_epoch = NUM_SAMPLES // FLAGS.batch_size
  epoch_time_list=[]
  test_accuracy_list = []
  for epoch in range(1, FLAGS.epochs + 1):
    np.random.seed(epoch)
    print(f'\n\n\n\n\n\n\n\n\n\n\n****** Starting epoch: {epoch}\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n')
    # Train the model for one epoch.
    start_time=time.time()
    lm_classifier.train(input_fn=train_input_fn, steps=steps_per_epoch)
    end_time=time.time()
    epoch_time_list.append(end_time-start_time)
    print(f'\n\n\n\n\n\n\n\n\n\n\n\n\n\nepoch {epoch} time: {end_time-start_time}')
    print('\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n******** Epoch completed\n\n\n\n\n\n\n\n\n\n\n\n')
    
      # Evaluate the model and print results
    eval_results = lm_classifier.evaluate(input_fn=eval_input_fn)
    test_accuracy = eval_results['accuracy']
    test_accuracy_list.append(test_accuracy)    
    print('\n\n\n\n\n\n\n\n\n\n\n\n************ Test accuracy after %d epochs is: %.3f\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n' % (epoch, test_accuracy))
    print('epoch time list: ',epoch_time_list)
    print('test_accuracy_list',test_accuracy_list)
    print('\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n')
    if FLAGS.dpsgd:
        if FLAGS.microbatches==FLAGS.batch_size:
            with open('RNN_DP_TF256.pickle', 'wb') as handle:
                pickle.dump(test_accuracy_list, handle)
        else:
            with open('RNN_DP_TF256_microbatch'+str(FLAGS.microbatches)+'.pickle', 'wb') as handle:
                pickle.dump(test_accuracy_list, handle)

if __name__ == '__main__':
  app.run(main)
