# Copyright 2025 The HuggingFace Team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# code from https://github.com/Deep-Agent/R1-V

import os
import textwrap
from collections import defaultdict
from typing import Any, Callable, Optional, Union

import torch
import torch.utils.data
import transformers
from datasets import Dataset, IterableDataset
from packaging import version
from transformers import (
    AriaForConditionalGeneration,
    AriaProcessor,
    AutoModelForCausalLM,
    AutoModelForSequenceClassification,
    AutoProcessor,
    AutoTokenizer,
    GenerationConfig,
    PreTrainedModel,
    PreTrainedTokenizerBase,
    Qwen2VLForConditionalGeneration,
    Qwen2_5_VLForConditionalGeneration,
    LlavaForConditionalGeneration,
    LlavaOnevisionForConditionalGeneration,
    Trainer,
    TrainerCallback,
    is_wandb_available,
)
from transformers.integrations.deepspeed import is_deepspeed_zero3_enabled
from transformers.utils import is_peft_available

from trl.data_utils import apply_chat_template, is_conversational, maybe_apply_chat_template
from trl.models import create_reference_model, prepare_deepspeed, unwrap_model_for_generation
from trl.trainer.grpo_config import GRPOConfig
from trl.trainer.utils import generate_model_card, get_comet_experiment_url

from PIL import Image
import openai
import base64
openai.api_key = os.getenv("OPENAI_API_KEY")

import copy
import re


if is_peft_available():
    from peft import PeftConfig, get_peft_model

if is_wandb_available():
    import wandb

# What we call a reward function is a callable that takes a list of prompts and completions and returns a list of
# rewards. When it's a string, it's a model ID, so it's loaded as a pretrained model.
RewardFunc = Union[str, PreTrainedModel, Callable[[list, list], list[float]]]

def encode_image(image_path):
    with open(image_path, "rb") as image_file:
        return base64.b64encode(image_file.read()).decode("utf-8")
    
def evaluate_with_gpt4(evaluation_template, question, good_answer, generated_answer, image_path=None):
    client = openai.OpenAI()
    prompt = evaluation_template.format(
        question=question,
        good_answer=good_answer,
        generated_answer=generated_answer
    )
    if image_path and any(image_path.lower().endswith(ext) for ext in ['.png', '.jpeg', '.jpg', '.gif', '.webp']):
        base64_image = encode_image(image_path)
        response = client.chat.completions.create(
            model="gpt-4o-mini-2024-07-18",  # You can use 'gpt-4' or 'gpt-4-turbo'
            messages=[
                {
                "role": "user",
                "content": [
                    {"type": "text", "text": prompt},
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": f"data:image/jpeg;base64,{base64_image}",
                        },
                    },
                ],
            }
            ],
            temperature=0.0,
            max_tokens=500
        )
    else:
        response = client.chat.completions.create(
            model="gpt-4o-mini-2024-07-18",  # You can use 'gpt-4' or 'gpt-4-turbo'
            messages=[
                {
                "role": "user",
                "content": [
                    {"type": "text", "text": prompt},
                ],
            }
            ],
            temperature=0.0,
            max_tokens=500
        )

    evaluation_result = response.choices[0].message.content.strip()
    return evaluation_result


def extract_final_decision(evaluation_text):
    match = re.search(r"Final Decision[:\s*]*\*{0,2}(\d)\*{0,2}", evaluation_text, re.IGNORECASE)
    return int(match.group(1)) if match else 0.5

def extract_final_decision_v8(model_response):
    response_clean = model_response.strip().lower()
    if "yes" in response_clean:
        return 1
    elif "no" in response_clean:
        return 0
    else:
        return 0.5

def extract_final_decision_v9(model_response):
    response_clean = model_response.strip().lower()
    if "yes" in response_clean:
        return 1
    else:
        return 0.5
    
def extract_final_decision_v10(model_response):
    response_clean = model_response.strip().lower()
    if "no" in response_clean:
        return 0
    else:
        return 0.5
    
def extract_final_decision_v11(model_response):
    pattern = r'\*\*Final Verdict\*\*:\s*(?:\*\*)?(Yes|No)(?:\*\*)?'
    match = re.search(pattern, model_response, re.IGNORECASE)
    if match:
        if match.group(1).lower() == "yes":
            return 1
        elif match.group(1).lower() == "no":
            return 0
    else:
        return 0.5

def check_multi_yn(winner_answer):
    if winner_answer in ["yes", "no"] or winner_answer in ["A", "B", "C", "D", "E", "F", "A.", "B.", "C.", "D.", "E.", "F."]:
        return True
    else:
        return False
    
def multi_yn_format_reward(completions, winner_answer, device):
    reward = torch.zeros(len(completions), device=device)
    for i, com in enumerate(completions):
        if winner_answer.replace(".", "").lower() in com.split()[0].replace(" ", "").lower():
            reward[i] = 2
    return reward


class Qwen2VLGRPOTrainer(Trainer):
    """
    Trainer for the Group Relative Policy Optimization (GRPO) method. This algorithm was initially proposed in the
    paper [DeepSeekMath: Pushing the Limits of Mathematical Reasoning in Open Language Models](https://huggingface.co/papers/2402.03300).

    Example:

    ```python
    from datasets import load_dataset
    from trl import GRPOTrainer

    dataset = load_dataset("trl-lib/tldr", split="train")

    trainer = GRPOTrainer(
        model="Qwen/Qwen2-0.5B-Instruct",
        reward_funcs="weqweasdas/RM-Gemma-2B",
        train_dataset=dataset,
    )

    trainer.train()
    ```

    Args:
        model (`Union[str, PreTrainedModel]`):
            Model to be trained. Can be either:

            - A string, being the *model id* of a pretrained model hosted inside a model repo on huggingface.co, or
              a path to a *directory* containing model weights saved using
              [`~transformers.PreTrainedModel.save_pretrained`], e.g., `'./my_model_directory/'`. The model is
              loaded using [`~transformers.AutoModelForCausalLM.from_pretrained`] with the keywork arguments
              in `args.model_init_kwargs`.
            - A [`~transformers.PreTrainedModel`] object. Only causal language models are supported.
        reward_funcs (`Union[RewardFunc, list[RewardFunc]]`):
            Reward functions to be used for computing the rewards. To compute the rewards, we call all the reward
            functions with the prompts and completions and sum the rewards. Can be either:

            - A single reward function, such as:
                - A string: The *model ID* of a pretrained model hosted inside a model repo on huggingface.co, or a
                path to a *directory* containing model weights saved using
                [`~transformers.PreTrainedModel.save_pretrained`], e.g., `'./my_model_directory/'`. The model is loaded
                using [`~transformers.AutoModelForSequenceClassification.from_pretrained`] with `num_labels=1` and the
                keyword arguments in `args.model_init_kwargs`.
                - A [`~transformers.PreTrainedModel`] object: Only sequence classification models are supported.
                - A custom reward function: The function is provided with the prompts and the generated completions,
                  plus any additional columns in the dataset. It should return a list of rewards. For more details, see
                  [Using a custom reward function](#using-a-custom-reward-function).
            - A list of reward functions, where each item can independently be any of the above types. Mixing different
            types within the list (e.g., a string model ID and a custom reward function) is allowed.
        args ([`GRPOConfig`], *optional*, defaults to `None`):
            Configuration for this trainer. If `None`, a default configuration is used.
        train_dataset ([`~datasets.Dataset`] or [`~datasets.IterableDataset`]):
            Dataset to use for training. It must include a column `"prompt"`. Any additional columns in the dataset is
            ignored. The format of the samples can be either:

            - [Standard](dataset_formats#standard): Each sample contains plain text.
            - [Conversational](dataset_formats#conversational): Each sample contains structured messages (e.g., role
              and content).
        eval_dataset ([`~datasets.Dataset`], [`~datasets.IterableDataset`] or `dict[str, Union[Dataset, IterableDataset]]`):
            Dataset to use for evaluation. It must meet the same requirements as `train_dataset`.
        processing_class ([`~transformers.PreTrainedTokenizerBase`], *optional*, defaults to `None`):
            Processing class used to process the data. The padding side must be set to "left". If `None`, the
            processing class is loaded from the model's name with [`~transformers.AutoTokenizer.from_pretrained`].
        reward_processing_classes (`Union[PreTrainedTokenizerBase, list[PreTrainedTokenizerBase]]`, *optional*, defaults to `None`):
            Processing classes corresponding to the reward functions specified in `reward_funcs`. Can be either:

            - A single processing class: Used when `reward_funcs` contains only one reward function.
            - A list of processing classes: Must match the order and length of the reward functions in `reward_funcs`.
            If set to `None`, or if an element of the list corresponding to a [`~transformers.PreTrainedModel`] is
            `None`, the tokenizer for the model is automatically loaded using [`~transformers.AutoTokenizer.from_pretrained`].
            For elements in `reward_funcs` that are custom reward functions (not [`~transformers.PreTrainedModel`]),
            the corresponding entries in `reward_processing_classes` are ignored.
        callbacks (list of [`~transformers.TrainerCallback`], *optional*, defaults to `None`):
            List of callbacks to customize the training loop. Will add those to the list of default callbacks
            detailed in [here](https://huggingface.co/docs/transformers/main_classes/callback).

            If you want to remove one of the default callbacks used, use the [`~transformers.Trainer.remove_callback`]
            method.
        optimizers (`tuple[torch.optim.Optimizer, torch.optim.lr_scheduler.LambdaLR]`, *optional*, defaults to `(None, None)`):
            A tuple containing the optimizer and the scheduler to use. Will default to an instance of [`AdamW`] on your
            model and a scheduler given by [`get_linear_schedule_with_warmup`] controlled by `args`.
        peft_config ([`~peft.PeftConfig`], *optional*, defaults to `None`):
            PEFT configuration used to wrap the model. If `None`, the model is not wrapped.
    """

    def __init__(
        self,
        model: Union[str, PreTrainedModel],
        reward_funcs: Union[RewardFunc, list[RewardFunc]],
        args: GRPOConfig = None,
        train_dataset: Optional[Union[Dataset, IterableDataset]] = None,
        eval_dataset: Optional[Union[Dataset, IterableDataset, dict[str, Union[Dataset, IterableDataset]]]] = None,
        processing_class: Optional[PreTrainedTokenizerBase] = None,
        reward_processing_classes: Optional[Union[PreTrainedTokenizerBase, list[PreTrainedTokenizerBase]]] = None,
        callbacks: Optional[list[TrainerCallback]] = None,
        optimizers: tuple[Optional[torch.optim.Optimizer], Optional[torch.optim.lr_scheduler.LambdaLR]] = (None, None),
        peft_config: Optional["PeftConfig"] = None,
        max_pixels: Optional[int] = 12845056,
        min_pixels: Optional[int] = 3136,
        Reward_eplison: Optional[float] = 0.1,
        Reward_version: Optional[str] = "v1",
        attn_implementation: str = "flash_attention_2",
    ):
        # Args
        if args is None:
            model_name = model if isinstance(model, str) else model.config._name_or_path
            model_name = model_name.split("/")[-1]
            args = GRPOConfig(f"{model_name}-GRPO")
        self.Reward_eplison = Reward_eplison
        self.Reward_version = Reward_version
        # Models
        # Trained model
        model_init_kwargs = args.model_init_kwargs or {}
        model_init_kwargs["attn_implementation"] = attn_implementation
        if isinstance(model, str):
            model_id = model
            self.model_id = model_id
            torch_dtype = model_init_kwargs.get("torch_dtype")
            if isinstance(torch_dtype, torch.dtype) or torch_dtype == "auto" or torch_dtype is None:
                pass  # torch_dtype is already a torch.dtype or "auto" or None
            elif isinstance(torch_dtype, str):  # it's a str, but not "auto"
                torch_dtype = getattr(torch, torch_dtype)
                model_init_kwargs["torch_dtype"] = torch_dtype
            else:
                raise ValueError(
                    "Invalid `torch_dtype` passed to `GRPOConfig`. Expected either 'auto' or a string representing "
                    f"a `torch.dtype` (e.g., 'float32'), but got {torch_dtype}."
                )
            # Disable caching if gradient checkpointing is enabled (not supported)
            model_init_kwargs["use_cache"] = (
                False if args.gradient_checkpointing else model_init_kwargs.get("use_cache")
            )
            if "Qwen2-VL" in model_id:
                model = Qwen2VLForConditionalGeneration.from_pretrained(model, **model_init_kwargs)
            elif "Qwen2.5-VL" in model_id:
                model = Qwen2_5_VLForConditionalGeneration.from_pretrained(model, **model_init_kwargs)
            elif "Aria" in model_id:
                model_init_kwargs.pop("use_cache")
                model = AriaForConditionalGeneration.from_pretrained(model, **model_init_kwargs)
            elif "llava-onevision" in model_id:
                model_init_kwargs.pop("use_cache")
                model = LlavaOnevisionForConditionalGeneration.from_pretrained(model, **model_init_kwargs)
            elif "llava-1.5" in model_id:
                model_init_kwargs.pop("use_cache")
                model = LlavaForConditionalGeneration.from_pretrained(model, **model_init_kwargs)
            else:
                model = AutoModelForCausalLM.from_pretrained(model, **model_init_kwargs)
        else:
            model_id = model.config._name_or_path
            if args.model_init_kwargs is not None:
                raise ValueError(
                    "You passed `model_init_kwargs` to the `GRPOConfig`, but your model is already instantiated. "
                    "This argument can only be used when the `model` argument is a string."
                )

        if peft_config is not None:
            model = get_peft_model(model, peft_config)

        # Reference model
        if is_deepspeed_zero3_enabled():
            if "Qwen2-VL" in model_id:
                self.ref_model = Qwen2VLForConditionalGeneration.from_pretrained(model_id, **model_init_kwargs)
            elif "Qwen2.5-VL" in model_id:
                self.ref_model = Qwen2_5_VLForConditionalGeneration.from_pretrained(model_id, **model_init_kwargs)
            elif "Aria" in model_id:
                self.ref_model = AriaForConditionalGeneration.from_pretrained(model_id, **model_init_kwargs)
            elif "llava-1.5" in model_id:
                self.ref_model = LlavaForConditionalGeneration.from_pretrained(model_id, **model_init_kwargs)
            elif "llava-onevision" in model_id:
                self.ref_model = LlavaOnevisionForConditionalGeneration.from_pretrained(model_id, **model_init_kwargs)
            else:
                self.ref_model = AutoModelForCausalLM.from_pretrained(model_id, **model_init_kwargs)
        elif peft_config is None:
            # If PEFT configuration is not provided, create a reference model based on the initial model.
            self.ref_model = create_reference_model(model)
        else:
            # If PEFT is used, the reference model is not needed since the adapter can be disabled
            # to revert to the initial model.
            self.ref_model = None

        # Processing class
        if processing_class is None:
            if "Qwen2-VL" in model_id or "Qwen2.5-VL" in model_id or "Aria" in model_id or "llava" in model_id:
                processing_class = AutoProcessor.from_pretrained(model_id)
                pad_token_id = processing_class.tokenizer.pad_token_id
                processing_class.pad_token_id = pad_token_id
                processing_class.eos_token_id = processing_class.tokenizer.eos_token_id
                if "Qwen" in model_id or "Qwen2.5-VL" in model_id:
                    processing_class.image_processor.max_pixels = max_pixels
                    processing_class.image_processor.min_pixels = min_pixels
            else:
                processing_class = AutoTokenizer.from_pretrained(model.config._name_or_path, padding_side="left")
                pad_token_id = processing_class.pad_token_id

        # Reward functions
        if not isinstance(reward_funcs, list):
            reward_funcs = [reward_funcs]
        for i, reward_func in enumerate(reward_funcs):
            if isinstance(reward_func, str):
                reward_funcs[i] = AutoModelForSequenceClassification.from_pretrained(
                    reward_func, num_labels=1, **model_init_kwargs
                )
        self.reward_funcs = reward_funcs

        # Reward processing class
        if reward_processing_classes is None:
            reward_processing_classes = [None] * len(reward_funcs)
        elif not isinstance(reward_processing_classes, list):
            reward_processing_classes = [reward_processing_classes]
        else:
            if len(reward_processing_classes) != len(reward_funcs):
                raise ValueError("The number of reward processing classes must match the number of reward functions.")

        for i, (reward_processing_class, reward_func) in enumerate(zip(reward_processing_classes, reward_funcs)):
            if isinstance(reward_func, PreTrainedModel):
                if reward_processing_class is None:
                    reward_processing_class = AutoTokenizer.from_pretrained(reward_func.config._name_or_path)
                if reward_processing_class.pad_token_id is None:
                    reward_processing_class.pad_token = reward_processing_class.eos_token
                # The reward model computes the reward for the latest non-padded token in the input sequence.
                # So it's important to set the pad token ID to the padding token ID of the processing class.
                reward_func.config.pad_token_id = reward_processing_class.pad_token_id
                reward_processing_classes[i] = reward_processing_class
        self.reward_processing_classes = reward_processing_classes

        # Data collator
        def data_collator(features):  # No data collation is needed in GRPO
            return features

        # Training arguments
        self.max_prompt_length = args.max_prompt_length
        self.max_completion_length = args.max_completion_length  # = |o_i| in the GRPO paper
        self.num_generations = args.num_generations  # = G in the GRPO paper
        self.generation_config = GenerationConfig(
            max_new_tokens=self.max_completion_length,
            do_sample=True,  
            temperature=1.0, # HACK
            num_return_sequences=self.num_generations,
            pad_token_id=pad_token_id,
        )
        self.beta = args.beta

        # The trainer estimates the number of FLOPs (floating-point operations) using the number of elements in the
        # input tensor associated with the key "input_ids". However, in GRPO, the sampled data does not include the
        # "input_ids" key. Instead, the available keys is "prompt". As a result, the trainer issues the warning:
        # "Could not estimate the number of tokens of the input, floating-point operations will not be computed." To
        # suppress this warning, we set the "estimate_tokens" key in the model's "warnings_issued" dictionary to True.
        # This acts as a flag to indicate that the warning has already been issued.
        model.warnings_issued["estimate_tokens"] = True

        # Initialize the metrics
        self._metrics = defaultdict(list)

        super().__init__(
            model=model,
            args=args,
            data_collator=data_collator,
            train_dataset=train_dataset,
            eval_dataset=eval_dataset,
            processing_class=processing_class,
            callbacks=callbacks,
            optimizers=optimizers,
        )

        # Gradient accumulation requires scaled loss. Normally, loss scaling in the parent class depends on whether the
        # model accepts loss-related kwargs. Since we compute our own loss, this check is irrelevant. We set
        # self.model_accepts_loss_kwargs to False to enable scaling.
        self.model_accepts_loss_kwargs = False

        if self.ref_model is not None:
            if self.is_deepspeed_enabled:
                self.ref_model = prepare_deepspeed(self.ref_model, self.accelerator)
            else:
                self.ref_model = self.accelerator.prepare_model(self.ref_model, evaluation_mode=True)

        for i, reward_func in enumerate(self.reward_funcs):
            if isinstance(reward_func, PreTrainedModel):
                self.reward_funcs[i] = self.accelerator.prepare_model(reward_func, evaluation_mode=True)

    def _set_signature_columns_if_needed(self):
        # If `self.args.remove_unused_columns` is True, non-signature columns are removed.
        # By default, this method sets `self._signature_columns` to the model's expected inputs.
        # In GRPOTrainer, we preprocess data, so using the model's signature columns doesn't work.
        # Instead, we set them to the columns expected by the `training_step` method, hence the override.
        if self._signature_columns is None:
            self._signature_columns = ["prompt"]


    # Get the per-token log probabilities for the completions for the model and the reference model
    def _get_per_token_logps(self, model, input_ids, **kwargs):
        logits = model(input_ids, **kwargs).logits  # (B, L, V)
        logits = logits[:, :-1, :]  # (B, L-1, V), exclude the last logit: it corresponds to the next token pred
        input_ids = input_ids[:, 1:]  # (B, L-1), exclude the first input ID since we don't have logits for it
        # Compute the log probabilities for the input tokens. Use a loop to reduce memory peak.
        per_token_logps = []
        for logits_row, input_ids_row in zip(logits, input_ids):
            log_probs = logits_row.log_softmax(dim=-1)
            token_log_prob = torch.gather(log_probs, dim=1, index=input_ids_row.unsqueeze(1)).squeeze(1)
            per_token_logps.append(token_log_prob)
        return torch.stack(per_token_logps)


    # Trainer "prepares" the inputs before calling `compute_loss`. It converts to tensor and move to device.
    # Since we preprocess the data in `compute_loss`, we need to override this method to skip this step.
    def _prepare_inputs(self, inputs: dict[str, Union[torch.Tensor, Any]]) -> dict[str, Union[torch.Tensor, Any]]:
        return inputs

    def compute_loss(self, model, inputs, return_outputs=False, num_items_in_batch=None):
        if return_outputs:
            raise ValueError("The GRPOTrainer does not support returning outputs")
        
        prompts = [x["prompt"] for x in inputs]
        images = [Image.open(x["image"]) if isinstance(x["image"], str) else x["image"] for x in inputs]
        if self.Reward_version in ["multi_yn1"]:
            inputs[0]["prompt"][0]["content"][1]["text"] = inputs[0]["prompt"][0]["content"][1]["text"].replace("Answer with the option's letter from the given choices directly.",
                                                             "Answer with the option's letter first and then the explanation.")
        if "llava-onevision" in self.model_id:
            images = [image.resize((384, 384)) for image in images]
        prompts_text = [maybe_apply_chat_template(example, self.processing_class)["prompt"] for example in inputs]
        prompt_inputs = self.processing_class(
            text=prompts_text,
            images=images,
            return_tensors="pt",
            padding=True,
            padding_side="left",
            add_special_tokens=False,
        )
        prompt_inputs = super()._prepare_inputs(prompt_inputs)

        prompt_ids, prompt_mask = prompt_inputs["input_ids"], prompt_inputs["attention_mask"]
        # pixel_values = prompt_inputs["pixel_values"]
        # image_grid_thw = prompt_inputs["image_grid_thw"]

        
        if self.max_prompt_length is not None:
            prompt_inputs["input_ids"] = prompt_inputs["input_ids"][:, -self.max_prompt_length :]
            prompt_inputs["attention_mask"] = prompt_inputs["attention_mask"][:, -self.max_prompt_length :]

        # Generate completions
        all_completions = []

        if "multi_yn" in self.Reward_version:
            if check_multi_yn(inputs[0]["winner_answer"]):
                with unwrap_model_for_generation(model, self.accelerator) as unwrapped_model:
                    prompt_completion_ids = unwrapped_model.generate(**prompt_inputs, generation_config=self.generation_config)
                    for ids in prompt_completion_ids:
                        all_completions.append(ids.unsqueeze(0))
            else: # keep the same code flow as multi_yn data, if not, the code will stack.
                with unwrap_model_for_generation(model, self.accelerator) as unwrapped_model:
                    cur_generation_config = copy.deepcopy(self.generation_config)
                    cur_generation_config.num_return_sequences = 1
                    cur_generation_config.max_new_tokens = 1
                    prompt_completion_ids = unwrapped_model.generate(**prompt_inputs, generation_config=cur_generation_config)
        else:
            if self.num_generations > 0:
                with unwrap_model_for_generation(model, self.accelerator) as unwrapped_model:
                    prompt_completion_ids = unwrapped_model.generate(**prompt_inputs, generation_config=self.generation_config)
                    all_completions.append(prompt_completion_ids)
        
        if "multi_yn" in self.Reward_version:
            if check_multi_yn(inputs[0]["winner_answer"]):
                pass
            else:
                loser_ids = self.processing_class.tokenizer(inputs[0]["loser_answer"], return_tensors='pt', add_special_tokens=False)["input_ids"]
                loser_ids = torch.cat([loser_ids, torch.tensor([self.processing_class.tokenizer.eos_token_id]).unsqueeze(0).repeat(loser_ids.size(0), 1)], dim=1)
                # for i in range(int(self.num_generations/2)):
                all_completions.append(torch.cat([prompt_inputs["input_ids"], loser_ids.to(model.device)], dim=1))
        else:
            loser_ids = self.processing_class.tokenizer(inputs[0]["loser_answer"], return_tensors='pt', add_special_tokens=False)["input_ids"]
            loser_ids = torch.cat([loser_ids, torch.tensor([self.processing_class.tokenizer.eos_token_id]).unsqueeze(0).repeat(loser_ids.size(0), 1)], dim=1)
            all_completions.append(torch.cat([prompt_inputs["input_ids"], loser_ids.to(model.device)], dim=1))

         # Stack all completions and pad if needed
        max_length = max(completion.size(1) for completion in all_completions)
        padded_completions = []

        for completion in all_completions:
            if completion.size(1) < max_length:
                padding = torch.full(
                    (completion.size(0), max_length - completion.size(1)),
                    self.processing_class.tokenizer.pad_token_id,
                    dtype=completion.dtype,
                    device=completion.device,
                )
                padded_completion = torch.cat([completion, padding], dim=1)
            else:
                padded_completion = completion
            padded_completions.append(padded_completion)

        # Stack all padded completions
        prompt_completion_ids = torch.cat(padded_completions, dim=0)

        if "multi_yn" in self.Reward_version:
            if check_multi_yn(inputs[0]["winner_answer"]):
                num_generations = self.num_generations  # num_generations must be multiples of 2
            else:
                # num_generations = self.num_generations
                num_generations = 2
        else:
            num_generations = self.num_generations+2
        prompt_mask = prompt_mask.repeat_interleave(num_generations, dim=0)
        prompt_length = prompt_inputs["input_ids"].size(1)
        completion_ids = prompt_completion_ids[:, prompt_length:]

        # Mask everything after the first EOS token
        is_eos = completion_ids == self.processing_class.eos_token_id
        device = self.accelerator.device
        eos_idx = torch.full((is_eos.size(0),), is_eos.size(1), dtype=torch.long, device=device)
        eos_idx[is_eos.any(dim=1)] = is_eos.int().argmax(dim=1)[is_eos.any(dim=1)]
        sequence_indices = torch.arange(is_eos.size(1), device=device).expand(is_eos.size(0), -1)
        completion_mask = (sequence_indices <= eos_idx.unsqueeze(1)).int()

        # Concatenate prompt_mask with completion_mask for logit computation
        prompt_inputs["attention_mask"] = torch.cat([prompt_mask, completion_mask], dim=1)  # (B*G, P+C)
        if "Qwen2-VL" in self.model_id or "Qwen2.5-VL" in self.model_id:
            # prompt_inputs.keys() = ['pixel_values', 'image_grid_thw], prompt_inputs["pixel_values"].shape = (11844, 1176)
            prompt_inputs["pixel_values"] = prompt_inputs["pixel_values"].unsqueeze(0).repeat(len(prompt_completion_ids), 1,1)
            prompt_inputs["image_grid_thw"] = prompt_inputs["image_grid_thw"].repeat_interleave(len(prompt_completion_ids), dim=0)
        elif "llava-1.5" in self.model_id:
            # prompt_inputs.keys() = ['pixel_values'], prompt_inputs["pixel_values"].shape = (1, 3, 336, 336)
            prompt_inputs["pixel_values"] = prompt_inputs["pixel_values"].repeat(len(prompt_completion_ids), 1, 1, 1)
        elif "llava-onevision" in self.model_id:
            # prompt_inputs.keys() = ['pixel_values'], prompt_inputs["pixel_values"].shape = (1, 3, 3, 336, 336)
            prompt_inputs["pixel_values"] = prompt_inputs["pixel_values"].repeat(len(prompt_completion_ids), 1, 1, 1, 1)
            prompt_inputs["image_sizes"] = prompt_inputs["image_sizes"].repeat(len(prompt_completion_ids), 1)

        prompt_inputs.pop("input_ids")
        per_token_logps = self._get_per_token_logps(model, prompt_completion_ids, **prompt_inputs)
        # Get rid of the prompt (-1 because of the shift done in get_per_token_logps)
        per_token_logps = per_token_logps[:, prompt_length - 1 :]

        with torch.inference_mode():
            if self.ref_model is not None:
                ref_per_token_logps = self._get_per_token_logps(self.ref_model, prompt_completion_ids, **prompt_inputs)
            else:
                with self.accelerator.unwrap_model(model).disable_adapter():
                    ref_per_token_logps = self._get_per_token_logps(model, prompt_completion_ids, **prompt_inputs)
        ref_per_token_logps = ref_per_token_logps[:, prompt_length - 1 :]

        # Compute the KL divergence between the model and the reference model
        per_token_kl = torch.exp(ref_per_token_logps - per_token_logps) - (ref_per_token_logps - per_token_logps) - 1

        # Decode the generated completions
        completions = self.processing_class.batch_decode(completion_ids, skip_special_tokens=True)

        # open a jsonl file and save the completions
        import json
        with open("data/appendix_example_completions.jsonl", "a") as f:
            example = {"question": inputs[0]["problem"], "image": inputs[0]["image"], "completion": completions}
            f.write(json.dumps(example) + "\n")
        print(example)
        

        # if is_conversational(inputs[0]):
        #     completions = [[{"role": "assistant", "content": completion}] for completion in completions]

        # Our image information gain reward. winner and loser answers's reward is 2 and 0 respectively.
        if "multi_yn" in self.Reward_version:
            if check_multi_yn(inputs[0]["winner_answer"]):
                rewards = multi_yn_format_reward(completions, inputs[0]["winner_answer"], device)
            else:
                rewards = torch.zeros(len(all_completions), device=device)
                rewards[:len(all_completions)//2] = 2   # winner answer's reward
        else:
            rewards = torch.zeros(len(all_completions), device=device)
            rewards[0] = 2   # winner answer's reward

        # if device.index == 1:
        #     print(inputs[0]["winner_answer"], completions, rewards,  '\n')
        
        prompt_inputs["pixel_values"] = torch.zeros_like(prompt_inputs["pixel_values"])
        with torch.inference_mode():
            with unwrap_model_for_generation(model, self.accelerator) as unwrapped_model:
                text_only_per_token_logps = self._get_per_token_logps(unwrapped_model, prompt_completion_ids, **prompt_inputs) 
                # text_only_per_token_logps = get_per_token_logps(unwrapped_model, prompt_completion_ids)
        text_only_per_token_logps = text_only_per_token_logps[:, prompt_length - 1 :]
        text_loss = -(text_only_per_token_logps.detach() * completion_mask).sum(dim=1) / completion_mask.sum(dim=1)
        image_text_loss = -(per_token_logps.detach() * completion_mask).sum(dim=1) / completion_mask.sum(dim=1)
        image_info_reward = (text_loss - image_text_loss).detach()
        scaled_image_info_reward = image_info_reward / (text_loss + 1e-4)

        # Compute grouped-wise rewards
        mean_grouped_rewards = rewards.view(-1, num_generations).mean(dim=1)
        std_grouped_rewards = rewards.view(-1, num_generations).std(dim=1)

        # Normalize the rewards to compute the advantages
        mean_grouped_rewards = mean_grouped_rewards.repeat_interleave(num_generations, dim=0)
        std_grouped_rewards = std_grouped_rewards.repeat_interleave(num_generations, dim=0)
        advantages = (rewards - mean_grouped_rewards) / (std_grouped_rewards + 1e-4)

        # x - x.detach() allows for preserving gradients from x
        per_token_loss = torch.exp(per_token_logps - per_token_logps.detach()) * advantages.unsqueeze(1)
        per_token_loss = -(per_token_loss - self.beta * per_token_kl)
        loss = ((per_token_loss * completion_mask).sum(dim=1) / completion_mask.sum(dim=1)).mean()

        # Log the metrics
        # evey metric calcualte mean first to deal with different batch size for different GPUs
        completion_length = self.accelerator.gather_for_metrics(completion_mask.sum(1).float().mean()).float().mean().item()
        self._metrics["completion_length"].append(completion_length)

        self._metrics["reward"].append(self.accelerator.gather_for_metrics(rewards.mean()).mean().item())
        self._metrics["reward_std"].append(self.accelerator.gather_for_metrics(std_grouped_rewards.mean()).mean().item())

        self._metrics["image_info_reward"].append(self.accelerator.gather_for_metrics(image_info_reward.mean()).mean().item())
        self._metrics["scaled_image_info_reward"].append(self.accelerator.gather_for_metrics(scaled_image_info_reward.mean()).mean().item())

        if "multi_yn" in self.Reward_version:
            if check_multi_yn(inputs[0]["winner_answer"]):
                if torch.sum(rewards==2) > 0:
                    winner_image_info_reward = image_info_reward[rewards==2].mean()
                    scaled_winner_image_info_reward = scaled_image_info_reward[rewards==2].mean()
                else:
                    winner_image_info_reward = torch.tensor(0, device=device, dtype=rewards.dtype)
                    scaled_winner_image_info_reward = torch.tensor(0, device=device, dtype=rewards.dtype)
                if torch.sum(rewards==0) > 0:
                    loser_image_info_reward = image_info_reward[rewards==0].mean()
                    scaled_loser_image_info_reward = scaled_image_info_reward[rewards==0].mean()
                else:
                    loser_image_info_reward = torch.tensor(0, device=device, dtype=rewards.dtype)
                    scaled_loser_image_info_reward = torch.tensor(0, device=device, dtype=rewards.dtype)
            else:
                winner_image_info_reward = image_info_reward[0]
                loser_image_info_reward = image_info_reward[-1]
                scaled_winner_image_info_reward = scaled_image_info_reward[0]
                scaled_loser_image_info_reward = scaled_image_info_reward[-1]
            self._metrics["winner_image_info_reward"].append(self.accelerator.gather_for_metrics(winner_image_info_reward).mean().item())
            self._metrics["loser_image_info_reward"].append(self.accelerator.gather_for_metrics(loser_image_info_reward).mean().item())
            self._metrics["scaled_winner_image_info_reward"].append(self.accelerator.gather_for_metrics(scaled_winner_image_info_reward).mean().item())
            self._metrics["scaled_loser_image_info_reward"].append(self.accelerator.gather_for_metrics(scaled_loser_image_info_reward).mean().item())
        else:
            if len(image_info_reward) > 2:
                self._metrics["generated_image_info_reward"].append(self.accelerator.gather_for_metrics(image_info_reward[1:-1]).mean().item())
                self._metrics["scaled_generated_image_info_reward"].append(self.accelerator.gather_for_metrics(scaled_image_info_reward[1:-1]).mean().item())
            self._metrics["winner_image_info_reward"].append(self.accelerator.gather_for_metrics(image_info_reward[0]).mean().item())
            self._metrics["scaled_winner_image_info_reward"].append(self.accelerator.gather_for_metrics(scaled_image_info_reward[0]).mean().item())
            self._metrics["loser_image_info_reward"].append(self.accelerator.gather_for_metrics(image_info_reward[-1]).mean().item())
            self._metrics["scaled_loser_image_info_reward"].append(self.accelerator.gather_for_metrics(scaled_image_info_reward[-1]).mean().item())

        mean_kl = ((per_token_kl * completion_mask).sum(dim=1) / completion_mask.sum(dim=1)).mean()
        self._metrics["kl"].append(self.accelerator.gather_for_metrics(mean_kl).mean().item())

        return loss

    def log(self, logs: dict[str, float], start_time: Optional[float] = None) -> None:
        metrics = {key: sum(val) / len(val) for key, val in self._metrics.items()}  # average the metrics
        logs = {**logs, **metrics}
        if version.parse(transformers.__version__) >= version.parse("4.47.0.dev0"):
            super().log(logs, start_time)
        else:  # transformers<=4.46
            super().log(logs)
        self._metrics.clear()

    def create_model_card(
        self,
        model_name: Optional[str] = None,
        dataset_name: Optional[str] = None,
        tags: Union[str, list[str], None] = None,
    ):
        """
        Creates a draft of a model card using the information available to the `Trainer`.

        Args:
            model_name (`str` or `None`, *optional*, defaults to `None`):
                Name of the model.
            dataset_name (`str` or `None`, *optional*, defaults to `None`):
                Name of the dataset used for training.
            tags (`str`, `list[str]` or `None`, *optional*, defaults to `None`):
                Tags to be associated with the model card.
        """
        if not self.is_world_process_zero():
            return

        if hasattr(self.model.config, "_name_or_path") and not os.path.isdir(self.model.config._name_or_path):
            base_model = self.model.config._name_or_path
        else:
            base_model = None

        tags = tags or []
        if isinstance(tags, str):
            tags = [tags]

        if hasattr(self.model.config, "unsloth_version"):
            tags.append("unsloth")
        citation = None

        model_card = generate_model_card(
            base_model=base_model,
            model_name=model_name,
            hub_model_id=self.hub_model_id,
            dataset_name=dataset_name,
            tags=tags,
            wandb_url=wandb.run.get_url() if is_wandb_available() and wandb.run is not None else None,
            comet_url=get_comet_experiment_url(),
            trainer_name="GRPO",
            trainer_citation=citation,
        )

        model_card.save(os.path.join(self.args.output_dir, "README.md"))