import os
from copy import deepcopy
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple
import re
import numpy as np
import torch
from transformers import AutoModelForCausalLM, AutoTokenizer

from ..rome.layer_stats import layer_stats
from ...util import nethook
from ...util.generate import generate_fast
from ...util.globals import *

from .compute_ks import compute_ks
from .compute_z import compute_z, get_module_input_output_at_words, find_fact_lookup_idx
from .TaskEdit_hparams import TaskEditHyperParams

# Cache variable(s)
CONTEXT_TEMPLATES_CACHE = None
COV_CACHE_all = {}
COV_CACHE_1={}
COV_CACHE_2={}
P1_loaded = False
P2_loaded = False
cache_c_new = False

def apply_TaskEdit_to_model(
    model: AutoModelForCausalLM,
    tok: AutoTokenizer,
    requests: List[Dict],
    hparams: TaskEditHyperParams,
    copy=False,
    return_orig_weights=False,
    cache_template: Optional[str] = None,
    keep_original_weight=False,
    **kwargs
) -> Dict[str, Tuple[torch.Tensor]]:
  #-> Tuple[AutoModelForCausalLM, Dict[str, Any]]:
    """
    Returns a model with the desired changes.
    :param copy: If true, will preserve the original model while creating a new one to edit.
        Note that you are responsible for deallocating the new model's memory to avoid leaks.
    :return: (1) the updated model, (2) an original copy of the weights that changed
    """
    #print('apply requests',requests)
    global P, P1, P2, P1_loaded,P2_loaded, cache_c, cache_c_new

    weights_copy = {}
    if copy:
        model = deepcopy(model)
    
    # Calculate the null-space projection matrix P
    # Please ensure that you have downloaded "null_space_project.pt" to the easyedit folder beforehand, or get the P by following calculation
    if not os.path.exists(hparams.P1_loc):
        print(f"The null-space projection matrix P does not exist and now calculate.")
        W_out = nethook.get_parameter(model, f"{hparams.rewrite_module_tmp.format(hparams.layers[-1])}.weight")
        if "llama" in hparams.model_name.lower() or "gpt-j-6b"  in hparams.model_name.lower():
            P1 = torch.zeros((len(hparams.layers), W_out.shape[1], W_out.shape[1]), device="cpu")
        elif "gpt2-xl" in hparams.model_name.lower():
            P1 = torch.zeros((len(hparams.layers), W_out.shape[0], W_out.shape[0]), device="cpu")
        else:
            P1 = torch.zeros((len(hparams.layers), 18944, 18944), device="cpu")
            #P2 = torch.zeros((len(hparams.layers), 18944, 18944), device="cpu")
        del W_out
        for i, layer in enumerate(hparams.layers):
            P1[i,:,:],_ = get_project(model, tok, layer, hparams)
        torch.save(P1, hparams.P1_loc)
        #torch.save(P2, P2_loc)
        P1_loaded = True
    else:
        P1 = torch.load(hparams.P1_loc)
        P1_loaded = True
    if not os.path.exists(hparams.P2_loc):
        print(f"The null-space projection matrix P does not exist and now calculate.")
        W_out = nethook.get_parameter(model, f"{hparams.rewrite_module_tmp.format(hparams.layers[-1])}.weight")
        if "llama" in hparams.model_name.lower() or "gpt-j-6b"  in hparams.model_name.lower():
            P2 = torch.zeros((len(hparams.layers), W_out.shape[1], W_out.shape[1]), device="cpu")
        elif "gpt2-xl" in hparams.model_name.lower():
            P2 = torch.zeros((len(hparams.layers), W_out.shape[0], W_out.shape[0]), device="cpu")
        else:
            P2 = torch.zeros((len(hparams.layers), 18944, 18944), device="cpu")
            #P2 = torch.zeros((len(hparams.layers), 18944, 18944), device="cpu")
        del W_out
        for i, layer in enumerate(hparams.layers):
            _,P2[i,:,:] = get_project(model, tok, layer, hparams)
        torch.save(P2, hparams.P2_loc)
        #torch.save(P2, P2_loc)
        P2_loaded = True
    else:
        P2 = torch.load(hparams.P2_loc)
        P2_loaded = True
    
    
    # Maintain the global variable cache_c to avoid redundant computations.
    # If this is the first calculation (i.e., cache_c_new == false), then initialize cache_c first
    if not cache_c_new:
        W_out = nethook.get_parameter(model, f"{hparams.rewrite_module_tmp.format(hparams.layers[-1])}.weight")
        if "llama" in hparams.model_name.lower() or "gpt-j-6b" in hparams.model_name.lower() or "qwen" in hparams.model_name.lower():
            cache_c = torch.zeros((len(hparams.layers), W_out.shape[1], W_out.shape[1]), device="cpu")
        elif "gpt2-xl" in hparams.model_name.lower():
            cache_c = torch.zeros((len(hparams.layers), W_out.shape[0], W_out.shape[0]), device="cpu")
        del W_out
        cache_c_new = True
    
    if len(re.findall(r'[\u4e00-\u9fa5]', requests[0]['prompt']))==0:
       #print('prompt',requests['prompt'])
       deltas = execute_AlphaEdit(model, tok, requests, hparams, P1, cache_template=cache_template)
    else:
       deltas = execute_AlphaEdit(model, tok, requests, hparams, P2, cache_template=cache_template)
    with torch.no_grad():
        for w_name, upd_m in deltas.items():
            upd_matrix = upd_m.to(f"cuda:{hparams.device}")
            w = nethook.get_parameter(model, w_name)
            upd_matrix = upd_matrix_match_shape(upd_matrix, w.shape)

            if return_orig_weights and w_name not in weights_copy:
                weights_copy[w_name] = w.detach().clone()
            w[...] += upd_matrix.float()
        
    # deltas_2 = execute_AlphaEdit(model, tok, requests_2, hparams, cache_template=cache_template)

    # with torch.no_grad():
    #     for w_name, upd_m in deltas_2.items():
    #        upd_matrix = upd_m.to(f"cuda:{hparams.device}")
    #        w = nethook.get_parameter(model, w_name)
    #        upd_matrix = upd_matrix_match_shape(upd_matrix, w.shape)
        
    #        w[...] += upd_matrix.float()
    print(f"New weights successfully inserted into {list(deltas.keys())}")

    return model, weights_copy


def execute_AlphaEdit(
    model: AutoModelForCausalLM,
    tok: AutoTokenizer,
    requests: List[Dict],
    hparams: TaskEditHyperParams,
    P_new: torch.Tensor,
    cache_template: Optional[str] = None,
) -> Dict[str, Tuple[torch.Tensor]]:
    """
    Executes the AlphaEdit update algorithm for the specified update at the specified layer
    Invariant: model at beginning of function == model at end of function
    """

    deltas = {}

    # Update target and print info
    requests = deepcopy(requests)
    for i, request in enumerate(requests):
        if request["target_new"][0] != " ":
            # Space required for correct tokenization
            requests[i]["target_new"] = " " + request["target_new"]
        if '{}' not in request['prompt']:
            assert request['subject'] in request['prompt'] or \
                   print(f"Subject:{request['subject']} do not exist in prompt: {request['prompt']}")
        requests[i]['prompt'] = requests[i]['prompt'].replace(requests[i]['subject'], '{}')
        print(
            f"Executing AlphaEdit algo for: "
            f"[{request['prompt']}] -> [{request['target_new']}]"
        )

    # Retrieve weights that user desires to change
    weights = {
        f"{hparams.rewrite_module_tmp.format(layer)}.weight": nethook.get_parameter(
            model, f"{hparams.rewrite_module_tmp.format(layer)}.weight"
        )
        for layer in hparams.layers
    }

    # Save old weights for future restoration
    weights_copy = {k: v.detach().clone() for k, v in weights.items()}

    # Compute z for final layer
    context_templates = get_context_templates(model, tok)
    z_layer = hparams.layers[-1]
    z_list = []
    #print('request',requests)
    for request in requests:
        # Retrieve k/v pair if already stored in cache
        cache_fname = (
            Path(
                str(cache_template).format(
                    z_layer, hparams.clamp_norm_factor, request["case_id"]
                )
            )
            if cache_template is not None
            else None
        )
        data_loaded = False
        print('cache_fname',cache_fname)
        if (
            cache_fname is not None  # Require cache template
            and cache_fname.exists()  # Cache file must exist
        ):
            try:
                data = np.load(cache_fname)
                z_list.append(torch.from_numpy(data["v_star"]).to(f"cuda:{hparams.device}"))
                data_loaded = True
            except Exception as e:
                print(f"Error reading cache file due to {e}. Recomputing...")

        # Compute k/v pair if not loaded from cache
        print('data_loaded',data_loaded)
        if not data_loaded:
            cur_z = compute_z(
                model,
                tok,
                request,
                hparams,
                z_layer,
                context_templates,
            )

            z_list.append(cur_z)

            if cache_fname is not None:
                cache_fname.parent.mkdir(exist_ok=True, parents=True)
                np.savez(
                    cache_fname,
                    **{
                        "v_star": cur_z.detach().cpu().numpy(),
                    },
                )
                print(f"Cached k/v pair at {cache_fname}")
    zs = torch.stack(z_list, dim=1)

    # Insert
    for i, layer in enumerate(hparams.layers):
        print(f"\n\nLAYER {layer}\n")

        # Get current model activations
        layer_ks = compute_ks(model, tok, requests, hparams, layer, context_templates).T
        print(f"Writing {layer_ks.size(1)} key/value pair(s) into layer {layer}")

        # Compute residual error
        cur_zs = get_module_input_output_at_words(
            model,
            tok,
            z_layer,
            context_templates=[request["prompt"] for request in requests],
            words=[request["subject"] for request in requests],
            module_template=hparams.layer_module_tmp,
            fact_token_strategy=hparams.fact_token,
        )[1].T
        targets = zs - cur_zs
        print("z error", torch.linalg.norm(targets, dim=0).mean())

        repeat_factor = (layer_ks.size(1) // targets.size(1))
        targets = targets.repeat_interleave(repeat_factor, dim=1)
        resid = targets / (len(hparams.layers) - i)  # Distribute residual across layers
        #print('P.shape',P.shape,'layer_ks',layer_ks.shape,'cache_c.shape',cache_c.shape)
        upd_matrix = torch.linalg.solve(
                P_new[i,:,:].to(f"cuda:{hparams.device}") @ (layer_ks.to(f"cuda:{hparams.device}") @ layer_ks.T.to(f"cuda:{hparams.device}") + cache_c[i,:,:].to(f"cuda:{hparams.device}")) + hparams.L2*torch.eye(layer_ks.shape[0], dtype=torch.float,device=f"cuda:{hparams.device}"),
                P_new[i,:,:].to(f"cuda:{hparams.device}") @ layer_ks.to(f"cuda:{hparams.device}") @ resid.T.to(f"cuda:{hparams.device}")
        )

        # Adjust update matrix shape
        weight_name = f"{hparams.rewrite_module_tmp.format(layer)}.weight"
        upd_matrix = upd_matrix_match_shape(upd_matrix, weights[weight_name].shape)

        print("orig norm", torch.linalg.norm(weights[weight_name]))
        print("upd norm", torch.linalg.norm(upd_matrix))

        # Update model weights and record desired changes in `delta` variable
        with torch.no_grad():
            weights[weight_name][...] = weights[weight_name] + upd_matrix.float()
            deltas[weight_name] = (
                upd_matrix.detach().cpu()
            )
        
        # Clear GPU memory
        #del U,S,cov
        for x in [layer_ks, cur_zs, targets]:
            x.cpu()
            del x
        torch.cuda.empty_cache()
    
    for i, layer in enumerate(hparams.layers):
        layer_ks = compute_ks(model, tok, requests, hparams, layer, context_templates).T
        cache_c[i,:,:] += layer_ks.cpu() @ layer_ks.cpu().T

    # Restore state of original model
    with torch.no_grad():
        for k, v in weights.items():
            v[...] = weights_copy[k]
    
    print(f"Deltas successfully computed for {list(weights.keys())}")

    return deltas


def get_cov(
    model: AutoModelForCausalLM,
    tok: AutoTokenizer,
    layer_name: str,
    mom2_dataset1: str,
    mom2_n_samples1: str,
    mom2_dataset2: str,
    mom2_n_samples2: str,
    mom2_dtype: str,
    inv: bool = False,
    force_recompute: bool = True,
    hparams=None,
) -> torch.Tensor:
    """
    Retrieves covariance statistics, then computes the algebraic inverse.
    Caches result for future use.
    """

    model_name = model.config._name_or_path.replace("/", "_")
    key = (model_name, layer_name)

    print(f"Retrieving covariance statistics for {model_name} @ {layer_name}.")
    if key not in COV_CACHE_1 or key not in COV_CACHE_2 or force_recompute:
        stat1 = layer_stats(
            model,
            tok,
            layer_name,
            hparams.stats_dir,
            mom2_dataset1,
            to_collect=["mom2"],
            sample_size=mom2_n_samples1,
            precision=mom2_dtype,
            hparams=hparams,
            force_recompute=True,
            rank=None
        )
        stat2 = layer_stats(
            model,
            tok,
            layer_name,
            hparams.stats_dir,
            mom2_dataset2,
            to_collect=["mom2"],
            sample_size=mom2_n_samples2,
            precision=mom2_dtype,
            hparams=hparams,
            force_recompute=force_recompute,
            rank=None
        )
        
        print('stat1',stat1)
        COV_CACHE_1[key] = stat1.mom2.moment().float().to("cpu")
        COV_CACHE_2[key] = stat2.mom2.moment().float().to("cpu")
        COV_CACHE_all[key]=torch.concat((stat1.mom2.moment().float(),stat2.mom2.moment().float()),dim=0).to("cpu")
        return (
        torch.inverse(COV_CACHE_all[key].to(f"cuda:{hparams.device}")) if inv else COV_CACHE_all[key].to(f"cuda:{hparams.device}"),
        torch.inverse(COV_CACHE_1[key].to(f"cuda:{hparams.device}")) if inv else COV_CACHE_1[key].to(f"cuda:{hparams.device}"),
        torch.inverse(COV_CACHE_2[key].to(f"cuda:{hparams.device}")) if inv else COV_CACHE_2[key].to(f"cuda:{hparams.device}"),
    )


def upd_matrix_match_shape(matrix: torch.Tensor, shape: torch.Size) -> torch.Tensor:
    """
    GPT-2 and GPT-J have transposed weight representations.
    Returns a matrix that matches the desired shape, else raises a ValueError
    """

    if matrix.shape == shape:
        return matrix
    elif matrix.T.shape == shape:
        return matrix.T
    else:
        raise ValueError(
            "Update matrix computed by AlphaEdit does not match original weight shape. "
            "Check for bugs in the code?"
        )


def get_context_templates(model, tok):
    global CONTEXT_TEMPLATES_CACHE

    if CONTEXT_TEMPLATES_CACHE is None:
        CONTEXT_TEMPLATES_CACHE = [["{}"]] + [
            [
                f.replace("{", " ").replace("}", " ") + ". {}"
                for f in generate_fast(
                    model,
                    tok,
                    ["The", "Therefore", "Because", "I", "You"],
                    n_gen_per_prompt=n_gen // 5,
                    max_out_len=length,
                )
            ]
            for length, n_gen in [(10, 5)]  # Be careful about changing this.
        ]
        print(f"Cached context templates {CONTEXT_TEMPLATES_CACHE}")

    return CONTEXT_TEMPLATES_CACHE

def get_project(model, tok, layer, hparams):
    force_recompute = False
    cov,cov1,cov2 = get_cov(
        model,
        tok,
        hparams.rewrite_module_tmp.format(layer),
        hparams.mom2_dataset1,
        hparams.mom2_n_samples1,
        hparams.mom2_dataset2,
        hparams.mom2_n_samples2
        if not force_recompute
        else hparams.mom2_n_samples // 10,
        hparams.mom2_dtype,
        force_recompute=force_recompute,
        hparams=hparams
    )
    U, S, _ = torch.linalg.svd(cov, full_matrices=False)
    U1, S1, _ = torch.linalg.svd(cov1, full_matrices=False)
    U2, S2, _ = torch.linalg.svd(cov2, full_matrices=False)
    threshold = hparams.nullspace_threshold
    small_singular_indices = (S < threshold).nonzero(as_tuple=True)[0]
    small_singular_indices1 = (S1 < threshold).nonzero(as_tuple=True)[0]
    small_singular_indices2 = (S2 < threshold).nonzero(as_tuple=True)[0]
    print('small_indices',len(small_singular_indices),' ','small_indices1',len(small_singular_indices1),' ','small_indices2',len(small_singular_indices2),'\n')
    #print('all_indices',len(S[0]))
    print('conflict_score',1-len(small_singular_indices)/min(len(small_singular_indices1),len(small_singular_indices2)))
    return U1[:, small_singular_indices1] @ U1[:, small_singular_indices1].T,U2[:, small_singular_indices2] @ U2[:, small_singular_indices2].T