#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import numpy as np

class PCA:
    def __init__(self, n_PCs=1):
        """
        Initialize parameters.

        Parameters:
        n_PCs : type, default=1
            The number of principal components to be used in the model. Must be smaller than or equal to X.shape[1].
        """
        self.n_PCs = n_PCs

    def fit(self, X):
        """
        Fit the model to the training data.

        Parameters:
        X : {array-like, sparse matrix}, shape (n_samples, n_features)
            Training data.
        Returns:
        self : object
            Returns self.
        """
        
        _, _, Vt = np.linalg.svd(X)
        
        self.V = Vt.T
        
        self._is_fitted = True

        return self

    def predict(self, X):
        """
        Predict reconstruction loss for samples in X.

        Parameters:
        X : {array-like, sparse matrix}, shape (n_samples, n_features)
            Samples.

        Returns:
        y_pred : array, shape (n_samples,)
            reconstruction loss
        """
        
        projection = np.dot(X, self.V[:,:self.n_PCs])
        
        reconstruction = np.dot(projection.reshape((-1,1)), np.reshape(self.V[:,:self.n_PCs].T, (1,-1)))
        
        y_pred = np.mean((reconstruction-X)**2, axis=1)
        
        return y_pred

    def fit_predict(self, X):
        """
        Fit the model to the training data and then predict on the same data.

        Parameters:
        X : {array-like, sparse matrix}, shape (n_samples, n_features)
            Training data.
        y : array-like, shape (n_samples,)
            Target values.

        Returns:
        y_pred : array, shape (n_samples,)
            Class labels predicted on the same data.
        """
        self.fit(X)
        return self.predict(X)
    

    def __sklearn_is_fitted__(self):
        """
        Check fitted status and return a Boolean value.
        """
        return hasattr(self, "_is_fitted") and self._is_fitted

