#!/usr/bin/env sh
# Execute real-data experiments (NInFEA and AVMNIST)
# Usage: 
#   ./02_paper_experiments/00_04_exec_realdata.sh [step]
# Steps:
#   all               Run the full sequence (prep -> train -> posttrain -> plot)
#   ninfea-prep       Download & prepare NInFEA data
#   ninfea-train-2mod Train NInFEA (2-modality script)
#   ninfea-train-3mod Train NInFEA (3-modality script)
#   avmnist-train     Train AVMNIST model (054_avmnist.py)
#   avmnist-posttrain Run AVMNIST post-training (055_avmnist_posttrain.py)
#   avmnist-plot      Create AVMNIST analysis figure (analysis/052_avmnist_plots.py)

set -euo pipefail

# environment hints
: ${PYTHON:=python}
: ${DATA_DIR:=01_data}

STEP=${1:-all}

run(){
  echo "\n>>> $@"
  sh -c "$@"
}

case "$STEP" in
  ninfea-prep)
    run "$PYTHON 02_paper_experiments/050_NInFEA_prep.py"
    ;;

  ninfea-train-2mod)
    run "$PYTHON 02_paper_experiments/051_ninfea_multimodal_training_2mods.py --epochs 5000 --gpu 0 --seed 42"
    ;;

  ninfea-train-3mod)
    run "$PYTHON 02_paper_experiments/051_ninfea_multimodal_training.py --epochs 5000 --gpu 0 --seed 42"
    ;;

  avmnist-train)
    run "$PYTHON 02_paper_experiments/054_avmnist.py --epochs 5000 --gpu 0 --seed 42"
    ;;

  avmnist-posttrain)
    run "$PYTHON 02_paper_experiments/055_avmnist_posttrain.py --epochs 1000 --gpu 0 --seed 42"
    ;;

  avmnist-plot)
    run "$PYTHON 02_paper_experiments/analysis/052_avmnist_plots.py --seed 42 --data_dir ${DATA_DIR}/processed/avmnist"
    ;;

  all)
    run "$PYTHON 02_paper_experiments/050_NInFEA_prep.py"
    run "$PYTHON 02_paper_experiments/051_ninfea_multimodal_training_2mods.py --epochs 1000 --gpu 0 --seed 42"
    run "$PYTHON 02_paper_experiments/051_ninfea_multimodal_training.py --epochs 1000 --gpu 0 --seed 42"
    run "$PYTHON 02_paper_experiments/054_avmnist.py --epochs 200 --gpu 0 --seed 42"
    run "$PYTHON 02_paper_experiments/055_avmnist_posttrain.py --epochs 50 --gpu 0 --seed 42"
    run "$PYTHON 02_paper_experiments/analysis/052_avmnist_plots.py --seed 42 --data_dir ${DATA_DIR}/processed/avmnist"
    ;;

  *)
    echo "Unknown step: $STEP" >&2
    echo "Available: all|ninfea-prep|ninfea-train-2mod|ninfea-train-3mod|avmnist-train|avmnist-posttrain|avmnist-plot" >&2
    exit 2
    ;;
esac

echo "\nAll requested steps for '$STEP' completed."
