import numpy as np
import json
import os
import sys
from pathlib import Path
project_root = Path(__file__).parent.parent.parent.absolute()
sys.path.append(str(project_root))
from src.data.simulation_by_variables import TabularDataSimulator

# Parameters from robustness tests
noise_variances = [0.0, 0.0001, 0.01, 0.1, 0.2, 0.3, 0.4, 0.5, 1.0, 2.0]  # update if you tested other values
n_samples = 10000
n_hidden_variables = 5
hidden_dist_type = "poisson"
data_dim = 50
latent_dim = 20
nonlinearity_level = 1
nonlinearity_type = "sigmoid"
hidden_connectivity = 0.4
data_sparsity = 0.0
noise_mean = 0.0
n_noise_components = 1

# Generate signal (no noise)
sim = TabularDataSimulator(
    n_samples=n_samples,
    n_hidden_variables=n_hidden_variables,
    hidden_dist_type=hidden_dist_type,
    data_dim=data_dim,
    nonlinearity_level=nonlinearity_level,
    nonlinearity_type=nonlinearity_type,
    hidden_connectivity=hidden_connectivity,
    data_sparsity=data_sparsity,
    noise_variance=0.0,
    noise_mean=noise_mean,
    n_noise_components=n_noise_components,
    random_seed=0,
)
signal, _, _ = sim.generate_data()
signal = np.array(signal)

# Calculate signal power
signal_power = np.mean(signal ** 2)

snr_dict = {}
for noise_var in noise_variances:
    # Generate noise
    noise = np.random.normal(loc=noise_mean, scale=np.sqrt(noise_var), size=signal.shape)
    noise_power = np.mean(noise ** 2)
    snr = signal_power / noise_power if noise_power > 0 else np.nan
    snr_dict[noise_var] = snr
    print(f"Noise variance: {noise_var}, SNR: {snr:.3f}")

# Save to JSON
out_path = "03_results/paper_results/unimodal/signal_to_noise_ratios.json"
os.makedirs(os.path.dirname(out_path), exist_ok=True)
with open(out_path, "w") as f:
    json.dump(snr_dict, f, indent=2)
print(f"Saved SNRs to {out_path}")
