import argparse
import copy


class Argument:
    def __init__(self, *args, _auto_default=None, **kwargs):
        self.args = args
        self.kwargs = kwargs
        self._auto_default = _auto_default
        if self._auto_default is None:
            self._auto_default = bool(self.kwargs.get("action") not in ["store_true", "store_false", argparse.BooleanOptionalAction])

    def copy(self):
        return Argument(*copy.copy(self.args),
                        _auto_default=self._auto_default,
                        **copy.copy(self.kwargs))

    def __getitem__(self, k):
        if isinstance(k, int):
            return self.args[k]
        elif isinstance(k, str):
            return self.kwargs[k]
        else:
            raise ValueError(f"Expected key to be int or string, got {type(k)}")

    def __setitem__(self, k, v):
        if isinstance(k, int):
            self.args[k] = v
        elif isinstance(k, str):
            self.kwargs[k] = v
        else:
            raise ValueError(f"Expected key to be int or string, got {type(k)}")

    @property
    def default(self):
        if self.kwargs.get("action") == "store_false":
            return True
        elif self.kwargs.get("action") == "store_true":
            return False
        else:
            return self.kwargs.get("default", None)

    def modify(self, **kwargs):
        for k, modifier in kwargs.items():
            if k == "_auto_default":
                prev_value = self._auto_default
            else:
                prev_value = self.__getitem__(k)

            if callable(modifier):
                new_value = modifier(prev_value)
            else:
                new_value = modifier

            if k == "_auto_default":
                self._auto_default = new_value
            else:
                self.__setitem__(k, new_value)

        return self


class ArgumentList:
    def __init__(self, _hooks=[], **kwargs):
        self.__kwargs = kwargs
        self._hooks = _hooks
        for k, v in self.__kwargs.items():
            if k.startswith("_"):
                raise AttributeError("An argument cannot start with a underscore")
            if not isinstance(v, Argument):
                raise AttributeError(f"Expected instance of Argument, found {type(v)}")

    def copy(self):
        return ArgumentList(_hooks=copy.copy(self._hooks),
                            **{k: v.copy() for k, v in self.__kwargs.items()})

    def __add__(self, other):
        if isinstance(other, ArgumentList):
            return ArgumentList(_hooks=self._hooks + other._hooks,
                                **(self.__kwargs | other.__kwargs))
        else:
            return NotImplemented

    def with_defaults(self, **kwargs):
        newlist = self.copy()
        for k, defaultvalue in kwargs.items():
            newlist.__kwargs[k]["default"] = defaultvalue
        return newlist

    def modify(self, arg, **kwargs):
        newlist = self.copy()
        newlist.__kwargs[arg].modify(**kwargs)
        return newlist

    def add_to_argparser(self, parser):
        for k, arg in self.__kwargs.items():
            args = copy.copy(arg.args)
            kwargs = copy.copy(arg.kwargs)
            if "dest" in kwargs:
                del kwargs["dest"]
            if arg._auto_default:
                kwargs["help"] = kwargs.get("help", "") + f" (default: {arg.default})"
            parser.add_argument(*args, dest=k, **kwargs)


class ArgumentParser(argparse.ArgumentParser):
    def __init__(self, *args, hooks=[], **kwargs):
        super().__init__(*args, **kwargs)
        self.hooks = hooks

    def __iadd__(self, other):
        if isinstance(other, ArgumentList):
            other.add_to_argparser(self)
            self.hooks += other._hooks
            return self
        else:
            return NotImplemented

    def parse_args(self, *args, hooks=[], **kwargs):
        pargs = super().parse_args(*args, **kwargs)
        for hook in self.hooks + hooks:
            hook(args=pargs)
        return pargs
