import argparse

def _rename(newname):
    def decorator(f):
        f.__name__ = newname
        return f
    return decorator

def _bound(arg, typeconv, lower_excl=None, lower_incl=None, upper_excl=None, upper_incl=None):
    i = typeconv(arg)
    if lower_excl is not None and not (i > lower_excl):
        raise argparse.ArgumentError
    elif lower_incl is not None and not (i >= lower_incl):
        raise argparse.ArgumentError
    if upper_excl is not None and not (i < upper_excl):
        raise argparse.ArgumentError
    elif upper_incl is not None and not (i <= upper_incl):
        raise argparse.ArgumentError
    return i

def _choice(arg, typeconv, choices):
    a = typeconv(arg)
    if a not in choices:
        raise argparse.ArgumentError
    return a

def _sep_list(arg, typeconv, delim=",", strip_values=False, trim_empty=False):
    vs = arg.split(delim)
    if strip_values:
        vs = [v.strip() for v in vs]
    if trim_empty:
        vs = [v for v in vs if len(v) > 0]
    conv_vs = [typeconv(v) for v in vs]
    return conv_vs


@_rename("positive integer")
def posint(arg):
    return _bound(arg, int, lower_excl=0)

@_rename("negative integer")
def negint(arg):
    return _bound(arg, int, upper_excl=0)

@_rename("non-negative integer")
def nonnegint(arg):
    return _bound(arg, int, lower_incl=0)


@_rename("positive float")
def posfloat(arg):
    return _bound(arg, float, lower_excl=0.0)

@_rename("negative float")
def negfloat(arg):
    return _bound(arg, float, upper_excl=0.0)

@_rename("non-negative float")
def nonnegfloat(arg):
    return _bound(arg, float, lower_incl=0)

@_rename("float in (0,1)")
def floatr_o0_1o(arg):
    return _bound(arg, float, lower_excl=0.0, upper_excl=1.0)

@_rename("float in (0,1]")
def floatr_o0_1c(arg):
    return _bound(arg, float, lower_excl=0.0, upper_incl=1.0)

@_rename("float in [0,1]")
def floatr_c0_1c(arg):
    return _bound(arg, float, lower_incl=0.0, upper_incl=1.0)


@_rename("comma separated non-negative integer list")
def nonnegint_commalist(arg):
    return _sep_list(arg, nonnegint, delim=",", strip_values=True)


@_rename("boolean")
def boolean(arg):
    V_TRUE = ["true", "t", "yes", "y", "1", "ja", "sure", "okay", "ok", "yay"]
    V_FALSE = ["false", "f", "no", "n", "0", "nej", "nay"]
    if arg.lower() in V_TRUE:
        return True
    elif arg.lower() in V_FALSE:
        return False
    else:
        raise argparse.ArgumentError
