"""
Grid drawing utilities for OpenGL
"""

from OpenGL.GL import *

def draw_xz(major_spacing=1.0, minors_per_major=3,
            max_x=5.0, min_x=None, max_z=5.0, min_z=None, y=0.0):
    """
    Draws a grid in X-Z plane
    """
    assert major_spacing > 0
    assert minors_per_major >= 0
    minor_spacing = major_spacing / (minors_per_major + 1)
    if min_x is None: min_x = -max_x
    if min_z is None: min_z = -max_z
    glBegin(GL_LINES)
    c_x = (max_x + min_x)/2
    c_z = (max_z + min_z)/2

    def gridline(x0,y,z0,x1=None,z1=None):
        if x1 is None: x1 = x0
        elif z1 is None: z1 = z0
        glVertex3fv((x0,y,z0))
        glVertex3fv((x1,y,z1))
    def major_gridline(*args, **kwargs):
        glColor3fv((0.6, 0.6, 0.6)) # light-gray color
        gridline(*args, **kwargs)
    def minor_gridline(*args, **kwargs):
        glColor3fv((0.4, 0.4, 0.4)) # light-gray color
        gridline(*args, **kwargs)

    major_x = c_x
    while major_x <= max_x:
        major_gridline(x0=major_x, y=y, z0=min_z, z1=max_x)
        minor_x = major_x + minor_spacing
        for _ in range(minors_per_major):
            if minor_x > max_x: break
            minor_gridline(x0=minor_x, y=y, z0=min_z, z1=max_x)
            minor_x += minor_spacing
        major_x += major_spacing
    major_x = c_x - major_spacing
    while major_x >= min_x:
        major_gridline(x0=major_x, y=y, z0=min_z, z1=max_x)
        minor_x = major_x + minor_spacing
        for _ in range(minors_per_major):
            minor_gridline(x0=minor_x, y=y, z0=min_z, z1=max_x)
            minor_x += minor_spacing
        major_x -= major_spacing
    major_z = c_x
    while major_z <= max_x:
        major_gridline(x0=min_x, x1=max_x, y=y, z0=major_z)
        minor_z = major_z + minor_spacing
        for _ in range(minors_per_major):
            if minor_z > max_z: break
            minor_gridline(x0=min_x, x1=max_x, y=y, z0=minor_z)
            minor_z += minor_spacing
        major_z += major_spacing
    major_z = c_x - major_spacing
    while major_z >= min_x:
        major_gridline(x0=min_x, x1=max_x, y=y, z0=major_z)
        minor_z = major_z + minor_spacing
        for _ in range(minors_per_major):
            minor_gridline(x0=min_x, x1=max_x, y=y, z0=minor_z)
            minor_z += minor_spacing
        major_z -= major_spacing
    glEnd()

