import torch
from torch.optim.optimizer import Optimizer, required


class AnchoredSGD(Optimizer):
    """Anchored SGD optimizer"""

    def __init__(self, params, lr=0.01, gamma=0.0, weight_decay=0.0):

        if lr is not required and lr < 0.0:
            raise ValueError("Invalid learning rate: {}".format(lr))

        defaults = dict(lr=lr, gamma=gamma, weight_decay=weight_decay)
        self.idx = 0

        super(AnchoredSGD, self).__init__(params, defaults)

        for group in self.param_groups:
            for p in group['params']:
                state = self.state[p]
                state['anchor'] = torch.zeros_like(p)
                
    def update_anchor(self, anchor):

        current_index = 0

        for group in self.param_groups:
            for p in group['params']:
                numel = p.data.numel()
                size = p.data.size()

                state = self.state[p]
                state['anchor'] = (anchor[current_index:current_index+numel].view(size)).clone()

                current_index += numel

    @torch.no_grad()
    def step(self, closure=None):
        """Performs a single optimization step.

        Arguments:
            closure (callable, optional): A closure that reevaluates the model
                and returns the loss.
        """
        loss = None
        if closure is not None:
            loss = closure()

 
        # internal sgd update
        for group in self.param_groups:
            #get the lr
            lr = group['lr']
            gamma = group['gamma']
            wd = group['weight_decay']

            for p in group['params']:
                if p.grad is None:
                    continue
                state = self.state[p]

                grad = p.grad.data
                if wd != 0.0:
                    grad += wd * p.data
                d_p = gamma*grad + (p.data - state['anchor'])
                p.data.add_(d_p, alpha=-lr)
        
        self.idx += 1
        return loss
