import torch
import torch.nn.functional as F
from torch_geometric.nn import GATConv, GCNConv, SAGEConv
import torch.nn as nn

    
class MPNNs(torch.nn.Module):
    def __init__(self, in_channels, hidden_channels, out_channels, local_layers=3, 
                 dropout=0.5, heads=1, pre_ln=False, pre_linear=False, res=False, ln=False, bn=False, jk=False, gnn='gcn'):
        super(MPNNs, self).__init__()

        self.gnn = gnn

        self.dropout = dropout
        self.pre_ln = pre_ln

        self.pre_linear = pre_linear
        self.res = res
        self.ln = ln
        self.bn = bn
        self.jk = jk
        
        self.h_lins = torch.nn.ModuleList()
        self.local_convs = torch.nn.ModuleList()
        self.lins = torch.nn.ModuleList()
        self.lns = torch.nn.ModuleList()
        self.bns = torch.nn.ModuleList()
        if self.pre_ln:
            self.pre_lns = torch.nn.ModuleList()

        self.lin_in = torch.nn.Linear(in_channels, hidden_channels)
        
        if not self.pre_linear:
            if gnn=='gat':
                self.local_convs.append(GATConv(in_channels, hidden_channels, heads=heads,
                    concat=True, add_self_loops=False, bias=False))
            elif gnn=='sage':
                self.local_convs.append(SAGEConv(in_channels, hidden_channels))
            else:
                self.local_convs.append(GCNConv(in_channels, hidden_channels,
                        cached=False, normalize=True))
            self.lins.append(torch.nn.Linear(in_channels, hidden_channels))
            self.lns.append(torch.nn.LayerNorm(hidden_channels))
            self.bns.append(torch.nn.BatchNorm1d(hidden_channels))
            if self.pre_ln:
                self.pre_lns.append(torch.nn.LayerNorm(in_channels))
            local_layers = local_layers - 1
            
        for _ in range(local_layers):
            if gnn=='gat':
                self.local_convs.append(GATConv(hidden_channels, hidden_channels, heads=heads,
                    concat=True, add_self_loops=False, bias=False))
            elif gnn=='sage':
                self.local_convs.append(SAGEConv(hidden_channels, hidden_channels))
            else:
                self.local_convs.append(GCNConv(hidden_channels, hidden_channels,
                        cached=False, normalize=True))
            self.lins.append(torch.nn.Linear(hidden_channels, hidden_channels))
            self.lns.append(torch.nn.LayerNorm(hidden_channels))
            self.bns.append(torch.nn.BatchNorm1d(hidden_channels))
            if self.pre_ln:
                self.pre_lns.append(torch.nn.LayerNorm(hidden_channels))
                
        self.pred_local = torch.nn.Linear(hidden_channels, out_channels)

    def reset_parameters(self):
        for local_conv in self.local_convs:
            local_conv.reset_parameters()
        for lin in self.lins:
            lin.reset_parameters()
        for ln in self.lns:
            ln.reset_parameters()
        for bn in self.bns:
            bn.reset_parameters()
        if self.pre_ln:
            for p_ln in self.pre_lns:
                p_ln.reset_parameters()
        self.lin_in.reset_parameters()
        self.pred_local.reset_parameters()


    def forward(self, x, edge_index):
        
        if self.pre_linear:
            x = self.lin_in(x)
            x = F.dropout(x, p=self.dropout, training=self.training)
        
        x_final = 0
        
        for i, local_conv in enumerate(self.local_convs):
            if self.res:
                x = local_conv(x, edge_index) + self.lins[i](x)
            else:
                x = local_conv(x, edge_index)
            if self.ln:
                x = self.lns[i](x)
            elif self.bn:
                x = self.bns[i](x)
            else:
                pass
            x = F.relu(x)
            x = F.dropout(x, p=self.dropout, training=self.training)
            if self.jk:
                x_final = x_final + x
            else:
                x_final = x

        x = self.pred_local(x_final)

        return x


    def ham_func(self, lr, HK1, HK2):
        """Update the list of parameters of the convolutions."""

        def ham(w, grad, lr, HK1, HK2):
            return w * torch.exp(-lr * (grad * torch.sign(w) * HK1 + (HK2 * (grad != 0).float())))
       
        if self.pre_linear:
            self.lin_in.weight.data = ham(self.lin_in.weight.data, self.lin_in.weight.grad, lr, HK1, HK2)
            if self.lin_in.bias is not None:
                self.lin_in.bias.data = ham(self.lin_in.bias.data, self.lin_in.bias.grad, lr, HK1, HK2)

        for i in range(len(self.local_convs)):
            if self.gnn == 'gat':
                if hasattr(self.local_convs[i], 'lin_l') and hasattr(self.local_convs[i], 'lin_r'):
                    self.local_convs[i].lin_l.weight.data = ham(self.local_convs[i].lin_l.weight.data, self.local_convs[i].lin_l.weight.grad, lr, HK1, HK2)
                    self.local_convs[i].lin_r.weight.data = ham(self.local_convs[i].lin_r.weight.data, self.local_convs[i].lin_r.weight.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_l.bias is not None:
                        self.local_convs[i].lin_l.bias.data = ham(self.local_convs[i].lin_l.bias.data, self.local_convs[i].lin_l.bias.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_r.bias is not None:
                        self.local_convs[i].lin_r.bias.data = ham(self.local_convs[i].lin_r.bias.data, self.local_convs[i].lin_r.bias.grad, lr, HK1, HK2)
                    self.local_convs[i].att.data = ham(self.local_convs[i].att.data, self.local_convs[i].att.grad, lr, HK1, HK2)
                elif hasattr(self.local_convs[i], 'lin') and hasattr(self.local_convs[i], 'att_src')\
                    and hasattr(self.local_convs[i], 'att_dst'):
                    self.local_convs[i].lin.weight.data = ham(self.local_convs[i].lin.weight.data, self.local_convs[i].lin.weight.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin.bias is not None:
                        self.local_convs[i].lin.bias.data = ham(self.local_convs[i].lin.bias.data, self.local_convs[i].lin.bias.grad, lr, HK1, HK2)
                    self.local_convs[i].att_src.data = ham(self.local_convs[i].att_src.data, self.local_convs[i].att_src.grad, lr, HK1, HK2)
                    self.local_convs[i].att_dst.data = ham(self.local_convs[i].att_dst.data, self.local_convs[i].att_dst.grad, lr, HK1, HK2)
                elif hasattr(self.local_convs[i], 'lin_src') and hasattr(self.local_convs[i], 'lin_dst') \
                    and hasattr(self.local_convs[i], 'att_src') and hasattr(self.local_convs[i], 'att_dst'):
                    self.local_convs[i].lin_src.weight.data = ham(self.local_convs[i].lin_src.weight.data, self.local_convs[i].lin_src.weight.grad, lr, HK1, HK2)
                    self.local_convs[i].lin_dst.weight.data = ham(self.local_convs[i].lin_dst.weight.data, self.local_convs[i].lin_dst.weight.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_src.bias is not None:
                        self.local_convs[i].lin_src.bias.data = ham(self.local_convs[i].lin_src.bias.data, self.local_convs[i].lin_src.bias.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_dst.bias is not None:
                        self.local_convs[i].lin_dst.bias.data = ham(self.local_convs[i].lin_dst.bias.data, self.local_convs[i].lin_dst.bias.grad, lr, HK1, HK2)
                    self.local_convs[i].att_src.data = ham(self.local_convs[i].att_src.data, self.local_convs[i].att_src.grad, lr, HK1, HK2)
                    self.local_convs[i].att_dst.data = ham(self.local_convs[i].att_dst.data, self.local_convs[i].att_dst.grad, lr, HK1, HK2)
                elif hasattr(self.local_convs[i], 'weight') and hasattr(self.local_convs[i], 'bias')\
                    and self.local_convs[i].weight is not None:
                    self.local_convs[i].weight.data = ham(self.local_convs[i].weight.data, self.local_convs[i].weight.grad, lr, HK1, HK2)
                    self.local_convs[i].bias.data = ham(self.local_convs[i].bias.data, self.local_convs[i].bias.grad, lr, HK1, HK2)
                    print('Warning: GAT without attention')
                    print(f"The keys are {self.local_convs[i].state_dict().keys()}")
                else:
                    print('Warning: GAT without parameters')
                    print(f"The keys are {self.local_convs[i].state_dict().keys()}")
            elif self.gnn == 'gcn':
                if hasattr(self.local_convs[i], 'lin') and self.local_convs[i].lin is not None:
                    self.local_convs[i].lin.weight.data = ham(self.local_convs[i].lin.weight.data, self.local_convs[i].lin.weight.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin.bias is not None:
                        self.local_convs[i].lin.bias.data = ham(self.local_convs[i].lin.bias.data, self.local_convs[i].lin.bias.grad, lr, HK1, HK2)
                elif hasattr(self.local_convs[i], 'weight') and hasattr(self.local_convs[i], 'bias')\
                    and self.local_convs[i].weight is not None:
                    self.local_convs[i].weight.data = ham(self.local_convs[i].weight.data, self.local_convs[i].weight.grad, lr, HK1, HK2)
                    self.local_convs[i].bias.data = ham(self.local_convs[i].bias.data, self.local_convs[i].bias.grad, lr, HK1, HK2)
            elif self.gnn == 'sage':
                if hasattr(self.local_convs[i], 'lin_l') and hasattr(self.local_convs[i], 'lin_r'):
                    self.local_convs[i].lin_l.weight.data = ham(self.local_convs[i].lin_l.weight.data, self.local_convs[i].lin_l.weight.grad, lr, HK1, HK2)
                    self.local_convs[i].lin_r.weight.data = ham(self.local_convs[i].lin_r.weight.data, self.local_convs[i].lin_r.weight.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_l.bias is not None:
                        self.local_convs[i].lin_l.bias.data = ham(self.local_convs[i].lin_l.bias.data, self.local_convs[i].lin_l.bias.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_r.bias is not None:
                        self.local_convs[i].lin_r.bias.data = ham(self.local_convs[i].lin_r.bias.data, self.local_convs[i].lin_r.bias.grad, lr, HK1, HK2)
                elif hasattr(self.local_convs[i], 'lin_self') and hasattr(self.local_convs[i], 'lin_neigh'):
                    self.local_convs[i].lin_self.weight.data = ham(self.local_convs[i].lin_self.weight.data, self.local_convs[i].lin_self.weight.grad, lr, HK1, HK2)
                    self.local_convs[i].lin_neigh.weight.data = ham(self.local_convs[i].lin_neigh.weight.data, self.local_convs[i].lin_neigh.weight.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_self.bias is not None:
                        self.local_convs[i].lin_self.bias.data = ham(self.local_convs[i].lin_self.bias.data, self.local_convs[i].lin_self.bias.grad, lr, HK1, HK2)
                    if self.local_convs[i].lin_neigh.bias is not None:
                        self.local_convs[i].lin_neigh.bias.data = ham(self.local_convs[i].lin_neigh.bias.data, self.local_convs[i].lin_neigh.bias.grad, lr, HK1, HK2)