import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.nn.utils.weight_norm as weightNorm

from gen_rl.policy.env_models import DIM_LATENT


def conv3x3(in_planes, out_planes, stride=1):
    return weightNorm(nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=True))


class TReLU(nn.Module):
    def __init__(self):
        super(TReLU, self).__init__()
        self.alpha = nn.Parameter(torch.FloatTensor(1), requires_grad=True)
        self.alpha.data.fill_(0)

    def forward(self, x):
        x = F.relu(x - self.alpha) + self.alpha
        return x


def cfg(depth):
    depth_lst = [18, 34, 50, 101, 152]
    assert (depth in depth_lst), "Error : Resnet depth should be either 18, 34, 50, 101, 152"
    cf_dict = {
        '18': (BasicBlock, [2, 2, 2, 2]),
        '34': (BasicBlock, [3, 4, 6, 3]),
        '50': (Bottleneck, [3, 4, 6, 3]),
        '101': (Bottleneck, [3, 4, 23, 3]),
        '152': (Bottleneck, [3, 8, 36, 3]),
    }

    return cf_dict[str(depth)]


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(in_planes, planes, stride)
        self.conv2 = conv3x3(planes, planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                weightNorm(nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=True)),
            )
        self.relu_1 = TReLU()
        self.relu_2 = TReLU()

    def forward(self, x):
        out = self.relu_1(self.conv1(x))
        out = self.conv2(out)
        out += self.shortcut(x)
        out = self.relu_2(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = weightNorm(nn.Conv2d(in_planes, planes, kernel_size=1, bias=True))
        self.conv2 = weightNorm(nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=True))
        self.conv3 = weightNorm(nn.Conv2d(planes, self.expansion * planes, kernel_size=1, bias=True))
        self.relu_1 = TReLU()
        self.relu_2 = TReLU()
        self.relu_3 = TReLU()

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                weightNorm(nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=True)),
            )

    def forward(self, x):
        out = self.relu_1(self.conv1(x))
        out = self.relu_2(self.conv2(out))
        out = self.conv3(out)
        out += self.shortcut(x)
        out = self.relu_3(out)

        return out


class ResNet_wobn(nn.Module):
    def __init__(self, num_inputs, depth, num_outputs, **kwargs):
        super(ResNet_wobn, self).__init__()
        self._args = kwargs["args"]
        self.in_planes = 64

        block, num_blocks = cfg(depth)

        # shared architecture
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=2)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=1 if self._args["paint_if_patch"] else 2)

        if self._args["if_use_act_val_fn"]:
            # ==== Action to Image component
            self.state_conv = conv3x3(num_inputs, 32, 2)  # 32 for each of action and canvas
            self.state_relu = TReLU()
            self.a2img_conv1 = weightNorm(nn.Conv2d(self._args["action_dim"] + 2, 64, 1, 1, 0))
            self.a2img_conv2 = weightNorm(nn.Conv2d(64, 64, 1, 1, 0))
            self.a2img_conv3 = weightNorm(nn.Conv2d(64, 32, 1, 1, 0))
            self.a2img_relu_1 = TReLU()
            self.a2img_relu_2 = TReLU()
            self.a2img_relu_3 = TReLU()

            self.coord = torch.zeros([1, 2, 64, 64])
            for i in range(64):
                for j in range(64):
                    self.coord[0, 0, i, j] = i / 63.
                    self.coord[0, 1, i, j] = j / 63.
                    self.coord = self.coord.to(self._args["device"])
        else:
            self.conv1 = conv3x3(num_inputs, 64, 2)
            self.relu_1 = TReLU()

        if self._args["paint_if_patch"]:
            self.conv4 = weightNorm(nn.Conv2d(512, 1, 1, 1, 0))  # output-dim = 64
            if self._args["policy_name"].lower() in ["td3", "sac"]:
                self.conv4_2 = weightNorm(nn.Conv2d(512, 1, 1, 1, 0))  # output-dim = 64
        else:
            _dim_cnn = 8192
            self.fc = nn.Linear(_dim_cnn, num_outputs)

            if self._args["policy_name"].lower() in ["td3", "sac"]:
                self.fc_2 = nn.Linear(_dim_cnn, num_outputs)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []

        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        return nn.Sequential(*layers)

    def a2img(self, x):
        tmp = self.coord.expand(x.shape[0], 2, 64, 64)
        x = x.repeat(64, 64, 1, 1).permute(2, 3, 0, 1)
        x = self.a2img_relu_1(self.a2img_conv1(torch.cat([x, tmp], 1)))
        x = self.a2img_relu_2(self.a2img_conv2(x))
        x = self.a2img_relu_3(self.a2img_conv3(x))
        return x

    def forward(self, x, z=None, _if_Q1=False):
        # import pudb; pudb.start()
        if self._args["if_use_act_val_fn"]:
            s, a = x
            a = self.a2img(a)
            s = self.state_relu(self.state_conv(s))
            x = torch.cat([s, a], 1)
        else:
            x = self.relu_1(self.conv1(x))

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        if self._args["paint_if_patch"]:
            q = self.conv4(x)
            q = q.view(q.size(0), 64)
        else:
            q = F.avg_pool2d(x, 4)
            q = x.view(q.size(0), -1)
            q = self.fc(q)

        if _if_Q1:
            return q
        if self._args["policy_name"].lower() in ["td3", "sac"]:
            if self._args["paint_if_patch"]:
                q2 = self.conv4_2(x)
                q2 = q.view(q2.size(0), 64)
            else:
                q2 = F.avg_pool2d(x, 4)
                q2 = x.view(q2.size(0), -1)
                q2 = self.fc_2(q2)
            return q, q2
        else:
            return q

    def Q1(self, x, z=None):
        return self.forward(x, z, _if_Q1=True)
