# Packed Ensembles

Code source of *Packed Ensembles for Efficient Uncertainty Estimation*. Two notebooks enable the training of a Packed Ensembles ResNet50 on CIFAR10 and CIFAR100 respectively.

## Installation

In order to install the package `pysemble` required to run the notebooks, it is advised to follow the steps below:

```sh
cd packed-ensembles
conda create -n <env>
conda activate <env>
conda install pytorch torchvision cudatoolkit=11.3 -c pytorch
pip install .
```

## Run Experiments


* Notebook `resnet50_cifar10.ipynb` allows you to train a Packed Ensembles of ResNet50 on CIFAR-10. We specified a random seed value (`seed=0`) and ensured the training to be deterministic. We report here the performance of the model on the test datasets.

    | Acc | NLL | ECE | AUPR | AUC | FPR-95 |
    |-----|-----|-----|------|-----|--------|
    |95.91%|0.1358|0.0084|97.67%|95.72%|13.28%|

* Notebook `resnet50_cifar100.ipynb` has the same purpose but with CIFAR-100. We report here the performance of the model on the test datasets (`seed=0`).

    | Acc | NLL | ECE | AUPR | AUC | FPR-95 |
    |-----|-----|-----|------|-----|--------|
    |80.70%|0.7213|0.0224|90.23%|82.22%|53.53%|

## Usage

`pysemble` package defines two novel layers `EnsembleConv2d` and `EnsembleLinear` used to replace usual `torch.Conv2d` and `torch.Linear` layers enabling flexible ensembling.

Consider the following network (from [Pytorch Tutorial](https://pytorch.org/tutorials/beginner/blitz/neural_networks_tutorial.html#define-the-network)):

```py
import torch
import torch.nn as nn
import torch.nn.functional as F

class LeNet(nn.Module):
    def __init__(self):
        super().__init__()
        # 1 input image channel, 6 output channels, 5x5 square convolution
        # kernel
        self.conv1 = nn.Conv2d(1, 6, 5)
        self.conv2 = nn.Conv2d(6, 16, 5)
        # an affine operation: y = Wx + b
        self.fc1 = nn.Linear(16 * 5 * 5, 120)  # 5*5 from image dimension
        self.fc2 = nn.Linear(120, 84)
        self.fc3 = nn.Linear(84, 10)

    def forward(self, x):
        # Max pooling over a (2, 2) window
        x = F.max_pool2d(F.relu(self.conv1(x)), (2, 2))
        # If the size is a square, you can specify with a single number
        x = F.max_pool2d(F.relu(self.conv2(x)), 2)
        x = torch.flatten(x, 1) # flatten all dimensions except the batch dimension
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = self.fc3(x)
        return x
```

To create a Packed Ensembles with the following parameters: $M=4\ ; \alpha=2\ ; \gamma=2$ from the previous network modify the code accordingly:

```py
from einops import rearrange
from pysemble import EnsembleConv2d, EnsembleLinear

class PackedLeNet(nn.Module):
    def __init__(self):
        super().__init__()
        self.M = 4
        self.alpha = 2
        self.gamma = 2

        # The first layer is still a Conv2d since we want all subnetworks to get the original image channels
        self.conv1 = nn.Conv2d(1, 6*self.alpha, 5)
        self.conv2 = EnsembleConv2d(6*self.alpha, 16*self.alpha, 5, n_estimators=self.M, groups=self.gamma)

        self.fc1 = EnsembleLinear(16 * 5 * 5 * self.alpha, 120 * self.alpha, n_estimators=self.M, groups=self.gamma)
        self.fc2 = EnsembleLinear(120 * self.alpha, 84 * self.alpha, n_estimators=self.M, groups=self.gamma)
        self.fc3 = EnsembleLinear(84 * self.alpha, 10 * self.M, n_estimators=self.M)
    
    def forward(self, x):
        # Max pooling over a (2, 2) window
        x = F.max_pool2d(F.relu(self.conv1(x)), (2, 2))
        # If the size is a square, you can specify with a single number
        x = F.max_pool2d(F.relu(self.conv2(x)), 2)
        x = rearrange(x, "e (m c) h w -> (m e) c h w", m=self.M)
        x = torch.flatten(x, 1) # flatten all dimensions except the batch dimension
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = self.fc3(x)
        return x
```
