"""ResNet in PyTorch.

Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385
"""
# fmt: off
from typing import List, Type, Union

import torch.nn as nn
import torch.nn.functional as F
from einops import rearrange

from .ensemble_layers import EnsembleConv2d, EnsembleLinear


# fmt: on
class BasicBlock(nn.Module):
    expansion = 1

    def __init__(
        self,
        in_planes: int,
        planes: int,
        stride: int = 1,
        n_estimators: int = 4,
        n_subgroups: int = 1,
    ):
        super(BasicBlock, self).__init__()

        # No subgroups for the first layer
        self.conv1 = EnsembleConv2d(
            in_planes,
            planes,
            kernel_size=3,
            n_estimators=n_estimators,
            stride=stride,
            padding=1,
            bias=False,
        )
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = EnsembleConv2d(
            planes,
            planes,
            kernel_size=3,
            n_estimators=n_estimators,
            stride=1,
            padding=1,
            groups=n_subgroups,
            bias=False,
        )
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                EnsembleConv2d(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    n_estimators=n_estimators,
                    stride=stride,
                    groups=n_subgroups,
                    bias=False,
                ),
                nn.BatchNorm2d(self.expansion * planes),
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1, n_estimators=4, n_subgroups=1):
        super(Bottleneck, self).__init__()

        # No subgroups for the first layer
        self.conv1 = EnsembleConv2d(
            in_planes, planes, kernel_size=1, n_estimators=n_estimators, bias=False,
        )
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = EnsembleConv2d(
            planes,
            planes,
            kernel_size=3,
            n_estimators=n_estimators,
            stride=stride,
            padding=1,
            groups=n_subgroups,
            bias=False,
        )
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = EnsembleConv2d(
            planes,
            self.expansion * planes,
            kernel_size=1,
            n_estimators=n_estimators,
            groups=n_subgroups,
            bias=False,
        )
        self.bn3 = nn.BatchNorm2d(self.expansion * planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                EnsembleConv2d(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    n_estimators=n_estimators,
                    stride=stride,
                    groups=n_subgroups,
                    bias=False,
                ),
                nn.BatchNorm2d(self.expansion * planes),
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(
        self,
        block: Type[Union[BasicBlock, Bottleneck]],
        num_blocks: List[int],
        n_estimators: int,
        n_subgroups: int = 1,
        coeff: int = 2,
        num_classes: int = 10,
    ):
        super(ResNet, self).__init__()

        self.n_estimators = n_estimators
        self.coeff = coeff
        self.in_planes = int(64 * coeff)
        if self.in_planes % self.n_estimators:
            self.in_planes += self.n_estimators - self.in_planes % self.n_estimators
        block_planes = self.in_planes

        self.conv1 = nn.Conv2d(
            in_channels=3,
            out_channels=block_planes,
            kernel_size=3,
            stride=1,
            padding=1,
            bias=False,
        )

        self.bn1 = nn.BatchNorm2d(block_planes)
        self.layer1 = self._make_layer(
            block,
            block_planes,
            num_blocks[0],
            stride=1,
            n_estimators=n_estimators,
            n_subgroups=n_subgroups,
        )
        self.layer2 = self._make_layer(
            block,
            block_planes * 2,
            num_blocks[1],
            stride=2,
            n_estimators=n_estimators,
            n_subgroups=n_subgroups,
        )
        self.layer3 = self._make_layer(
            block,
            block_planes * 4,
            num_blocks[2],
            stride=2,
            n_estimators=n_estimators,
            n_subgroups=n_subgroups,
        )
        self.layer4 = self._make_layer(
            block,
            block_planes * 8,
            num_blocks[3],
            stride=2,
            n_estimators=n_estimators,
            n_subgroups=n_subgroups,
        )

        self.pool = nn.AdaptiveAvgPool2d(output_size=1)
        self.flatten = nn.Flatten(1)

        self.linear = EnsembleLinear(
            block_planes * 8 * block.expansion,
            num_classes * n_estimators,
            n_estimators,
        )

    def _make_layer(
        self, block, planes, num_blocks, stride, n_estimators, n_subgroups,
    ):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(
                block(self.in_planes, planes, stride, n_estimators, n_subgroups)
            )
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = rearrange(out, "e (m c) h w -> (m e) c h w", m=self.n_estimators)
        out = self.pool(out)
        out = self.flatten(out)
        out = self.linear(out)
        return out


def ResNet18_GrE(
    n_estimators: int, augmentation: int, n_subgroups: int, num_classes: int,
) -> ResNet:
    return ResNet(
        BasicBlock,
        [2, 2, 2, 2],
        n_estimators,
        n_subgroups,
        coeff=augmentation,
        num_classes=num_classes,
    )


def ResNet50_GrE(
    n_estimators: int, augmentation: int, n_subgroups: int, num_classes: int,
):
    return ResNet(
        Bottleneck,
        [3, 4, 6, 3],
        n_estimators,
        n_subgroups,
        coeff=augmentation,
        num_classes=num_classes,
    )
