import numpy as np
from openai import OpenAI
import re
import networkx as nx
import json
import copy
from google.genai import types

import utils
from utils import to_DeNF, is_number, decimal_eval
from utils import remove_last_zero
# import pandas as pd

knowledge_value_enum = {'True': 1, 'False': 0, 'Unknown': 2}
Three_value_set = ['True', 'TRUE', 'true', 'False', 'FALSE', 'false', 'Unknown', 'UNKNOWN', 'unknown']
Boolean_value_set = ['True', 'TRUE', 'true', 'False', 'FALSE', 'false']

class FCGraph:
    def __init__(self, domain, objects, interpretations, esl_single_model, args, prompt_temp_config, if_print=False):
        # paras for computation graph
        self.graph = nx.DiGraph()
        self.domain = domain
        self.new_inferred_node = set()
        self.new_inferred_node_rule_dict = {}
        self.if_print = if_print
        self.objects = objects
        self.inter_level = args.inter_level
        self.prompt_temp_config = prompt_temp_config

        self.interpretations = interpretations
        self.variables = esl_single_model[0]["variables"]
        self.predicates = esl_single_model[0]["predicates"]
        self.rule_set = esl_single_model[0]["rules"]
        self.knowledge_dict = {} # each knowledge: true, false (init-knowledge + preception_generated for inter_level=1)
        self.knowledge_prop_mapping = {}  # each knowledge -- unique proposition
        self.pre_assign = []  # pandas structure list for each raw_rule
        self.prop_dict = {}  # uniqueness, prop with Boolean values: 0: False, 1: True, 2: Unknown
        self.prop_rule_set = set()
        self.error_info = ''
        self.proposition_set = set()

    def init_perpcetion_model(self, model):
        self.perception_model = model

    def init_target_model(self, model):
        self.target_model = model
    def set_outFile(self, file):
        self.print_file = file

    def init(self):
        try:
            assert self.knowledge_init() == 0
            assert self.rule_init() == 0
            assert self.graph_init_without_unk() == 0
        except:
            return 1

    # the number argument for each knowledge use the clean number
    def knowledge_init(self):
        # parse interpretation:
        for inter in self.interpretations:
            pattern = r"(\w+)\((.*?)\)\s*=\s*(\w+)"
            match = re.findall(pattern, inter)
            # print(match)
            try:
                assert len(match) == 1
            except:
                error_info = f"\n!!! Type-2 Error in Stage-2: Abstraction result format error by the perception LLM (line-69: knowledge_init: match error).\n\nFailure Type = 2, Failure Stage = 2"
                self.error_info = error_info
                print(error_info)
                return 1

            inter_argus_raw = [i.strip() for i in match[0][1].split(",")]
            inter_argus = [remove_last_zero(s) if re.fullmatch(r'\d+(\.\d+)?', s) else s for s in inter_argus_raw]

            inter_pred_name = match[0][0] + "_" + str(len(inter_argus))
            inter_value = match[0][2].strip()
            try:
                assert inter_value in Three_value_set
            except:
                error_info = f"\n!!! Type-2 Error in Stage-2: Abstraction result format error by the perception LLM (line-81: knowledge_init). \n\tInvalid value assignment of knowledge for the predicate {inter_pred_name}({inter_argus}) with value {inter_value}.\n\nFailure Type = 2, Failure Stage = 2"
                self.error_info = error_info
                print(error_info)
                return 1

            if inter_pred_name not in self.knowledge_dict:
                self.knowledge_dict[inter_pred_name] = {}
            if tuple(inter_argus) not in self.knowledge_dict[inter_pred_name] and inter_value in Boolean_value_set: # same
                self.knowledge_dict[inter_pred_name][tuple(inter_argus)] = inter_value

            else:
                if tuple(inter_argus) in self.knowledge_dict[inter_pred_name]:
                    try:
                        assert self.knowledge_dict[inter_pred_name][tuple(inter_argus)] == inter_value
                    except:
                        error_info = f"\n!!! Type-2 Error in Stage-2: Abstraction result format error by the perception LLM (line-93: knowledge_init). \n\tConflict value of knowledge for the predicate {inter_pred_name} with arguments {inter_argus}.\n\nFailure Type = 2, Failure Stage = 2"
                        self.error_info = error_info
                        print(error_info)
                        return 1

        fprint_str = '\nThe original knowledge_dict is:'
        print("\nThe original knowledge_dict is: ")
        for dict_ele in self.knowledge_dict:
            print(f" \t {dict_ele}: {self.knowledge_dict[dict_ele]}\n")
            fprint_str = f"{fprint_str}\n\t {dict_ele}: {self.knowledge_dict[dict_ele]}"

        if self.if_print:
            self.print_file.write(fprint_str)

        self.raw_knowledge_dict = copy.deepcopy(self.knowledge_dict)

        return 0

    # propositionalize the rule set
    # Todo: DeNF first then obtain the proposition form, here we assume that the input rule are already in this DeNF form
    def rule_init(self):
        for rule in self.rule_set:
            match = re.findall(r'([~A-Z][a-zA-Z_]*)\((.*?)\)', rule)
            raw_rule = []
            for pred, args_str in match:
                args = args_str.replace(' ', '').split(',') if args_str else []
                pred = pred + '_' + str(len(args))
                raw_rule.append((pred, args))
            if_prop = ("True =>" in rule)
            prop_rule_list = self.propositionalize_rule(raw_rule, if_prop)
            try:
                prop_rule_list != 1
            except:
                return 1
            if "True =>" in rule:
                for each_rule in prop_rule_list:
                    self.proposition_set.add(' & '.join(each_rule))
            elif "=> False" in rule:
                assert("We currently do not support => False")
                exit(0)
            else:
                for each_rule in prop_rule_list:
                    prop_rule_str = ' & '.join(each_rule[:-1]) + ' => ' + each_rule[-1]
                    self.prop_rule_set.add(prop_rule_str)

        if self.if_print:
            print("\nThe knowledge_prop_mapping is: ")
            for dict_ele in self.knowledge_prop_mapping:
                print('\t', dict_ele, ":", self.knowledge_prop_mapping[dict_ele])

        return 0

    def check_var_consistency(self, var_mapping):
        """this is designed for math currently, for consideration of preserved math operators"""
        unique_var = set()
        unique_var_valued = set()
        combined_var = {}
        preserved_number_var = set() # currently we preserve number_var
        for key in var_mapping:
            # if is_number(key):
            #     continue
            clean_var = re.findall(r'\b[a-zA-Z_]\w*\b', key)
            if len(clean_var) == 1:
                unique_var.add(clean_var[0])
                unique_var_valued.update(i for i in clean_var if var_mapping[i] is not None)
            elif len(clean_var) == 0:
                preserved_number_var.add(key)
            else:
                combined_var[key] = clean_var

        unique_var.update(preserved_number_var)
        unique_var_valued.update(preserved_number_var)
        new_var_mapping = {i: var_mapping[i] for i in unique_var}

        for key in combined_var:
            var4compute = set(combined_var[key])
            value = key

            for sub_var in var4compute:
                if sub_var not in unique_var_valued:
                    continue
                value = re.sub(rf'\b{sub_var}\b', var_mapping[sub_var], value)


            new_var_mapping[key] = None

            try:
                if var4compute.issubset(unique_var_valued):
                    if self.domain == 'comp' or self.domain == 'ineq':
                        new_var_mapping[key] = remove_last_zero(str(decimal_eval(value)))
                    else:
                        new_var_mapping[key] = value
            except:
                new_var_mapping[key] = value.strip()

        return new_var_mapping

    def check_real_var_mapping(self,raw_rule ,var_mapping):
        """

        :param raw_rule: [('IsLarger_2', ['x', 'xB']), ('~Neg_1', ['xA']), ('IsLarger_2', ['x*xA', 'xB*xA'])]
        :param var_mapping: {'x': '9.11', 'x*xA': None, 'xA': None, 'xB': '9.11', 'xB*xA': None}
        :return: False if fail to find the value in the knowledge map for the full-instantiated predicate
        """
        for predicate, vars_tuple in raw_rule[:-1]:
            free_var = set()
            for var in vars_tuple:
                clean_var = re.findall(r'\b[a-zA-Z_]\w*\b', var)
                try:
                    free_var.update([x for x in clean_var if var_mapping[x] == None])
                except:
                    self.error_info=f"\n!!! Type-3 Error in Stage-3: Incorrect usage of variable. Please use the free variables before combined variable appeared.\n\nFailure Type = 9, Failure Stage = 3"
                    print(self.error_info)
                    return False, 1

            if_neg = predicate[0]=='~'
            if if_neg:
                predicate_counter = predicate[1:]
                neg_value = ['']
            else:
                predicate_counter = '~'+predicate

            if len(free_var) == 0:
                counter = 0
                if predicate in self.knowledge_dict:
                    vars_tuple_value = []
                    for var in vars_tuple:
                        vars_tuple_value.append(var_mapping[var])
                    vars_tuple_value = tuple(vars_tuple_value)
                    check_res = vars_tuple_value in self.knowledge_dict[predicate]

                    if not check_res or (vars_tuple_value in self.knowledge_dict[predicate] and self.knowledge_dict[predicate][vars_tuple_value] in ['False', 'FALSE',
                                                                                             'false']):
                        return False, 0

                    else:
                        counter += 1

                if counter == 0 and predicate_counter in self.knowledge_dict:
                    vars_tuple_value = []
                    for var in vars_tuple:
                        vars_tuple_value.append(var_mapping[var])
                    vars_tuple_value = tuple(vars_tuple_value)
                    check_res = vars_tuple_value in self.knowledge_dict[predicate_counter]
                    if not check_res or (vars_tuple_value in self.knowledge_dict[predicate_counter] and self.knowledge_dict[predicate_counter][vars_tuple_value] in ['True', 'TRUE',
                                                                                             'true']):
                        return False, 0

        return True, 0


    def generate_rules(self, var_index, unique_vars, var_mapping, prop_rule_list, raw_rule, if_prop):
        """ Recursively assign values to variables and generate rules """
        try:
            assert prop_rule_list != 1
        except:
            return 1

        if var_index >= len(unique_vars):
            prop_rule = []

            propagated_var_mapping = self.check_var_consistency(var_mapping)

            if propagated_var_mapping == 1:
                return 1

            if_real_mapping, check_status = self.check_real_var_mapping(raw_rule, propagated_var_mapping)

            if check_status == 1:
                return 1
            if not if_real_mapping:
                return prop_rule_list

            if self.inter_level >= 2:
                res = self.fulfill_var_mapping(var_mapping, propagated_var_mapping, raw_rule)
                try:
                    assert res == 0
                except:
                    return 1

            for predicate, vars_tuple in raw_rule:
                free_var = set()
                for var in vars_tuple:
                    clean_var = re.findall(r'\b[a-zA-Z_]\w*\b', var)
                    free_var.update([x for x in clean_var if propagated_var_mapping[x] == None])

                knowledge = f"{predicate}({', '.join(propagated_var_mapping[var] if propagated_var_mapping[var] is not None else var for var in vars_tuple)})"

                knowledge_value = 'Unknown'
                if_neg = predicate[0] == '~'
                if if_neg:
                    clean_predicate = predicate[1:]
                    clean_knowledge_value = ''
                    i=0
                    if clean_predicate in self.knowledge_dict:
                        match = re.findall(r"\((.*?)\)", knowledge)
                        try:
                            assert len(match) == 1
                        except:
                            error_info = f"\n!!! Type-2 Error in Stage-2/Stage-3: Abstraction/Generation result format error by the perception LLM (line-226: generate_rules). \n\tThe reason may be there exist brackets in the arguments for the knowledge {knowledge}.\n\nFailure Type = 2, Failure Stage = 2"
                            self.error_info = error_info
                            print(error_info)
                            return 1

                        tuple_key = tuple([mat.strip() for mat in match[0].split(',')])

                        if tuple_key in self.knowledge_dict[clean_predicate]:
                            i+=1
                            clean_knowledge_value = self.knowledge_dict[clean_predicate][tuple_key]
                            knowledge_value = utils.knowledge_value_neg(clean_knowledge_value)

                    if predicate in self.knowledge_dict:
                        match = re.findall(r"\((.*?)\)", knowledge)
                        try:
                            assert len(match) == 1
                        except:
                            error_info = f"\n!!! Type-2 Error in Stage-2/Stage-3: Abstraction/Generation result format error by the perception LLM (line-242: generate_rules). \n\tThe reason may be there exist brackets in the arguments for the knowledge {knowledge}.\n\nFailure Type = 2, Failure Stage = 2"
                            self.error_info = error_info
                            print(error_info)
                            return 1

                        tuple_key = tuple([mat.strip() for mat in match[0].split(',')])
                        if tuple_key in self.knowledge_dict[predicate]:
                            i+=1
                            knowledge_value = self.knowledge_dict[predicate][tuple_key]
                    if i==2:
                        try:
                            assert utils.knowledge_value_neg(clean_knowledge_value) == knowledge_value
                        except:
                            error_info = f"\n!!! Type-2 Error in Stage-2: Abstraction result format error by the perception LLM (line-254: generate_rules). \n\tWrong interpretation value for predicates {clean_predicate} and {predicate}. Only Unknown, True, False are allowed. Given clean: {clean_knowledge_value} and non-clean {knowledge_value}\n\nFailure Type = 2, Failure Stage = 2"
                            self.error_info = error_info
                            print(error_info)
                            return 1
                else:
                    predicate_neg = '~'+predicate
                    knowledge_value_neg = ''
                    i=0
                    if predicate in self.knowledge_dict:
                        match = re.findall(r"\((.*?)\)", knowledge)
                        try:
                            assert len(match) == 1
                        except:
                            error_info = f"\n!!! Type-2 Error in Stage-2/Stage-3: Abstraction/Generation result format error by the perception LLM (line-266: generate_rules). \n\tThe reason may be there exist brackets in the arguments for the knowledge {knowledge}.\n\nFailure Type = 2, Failure Stage = 2"
                            self.error_info = error_info
                            print(error_info)
                            return 1

                        tuple_key = tuple([mat.strip() for mat in match[0].split(',')])

                        if tuple_key in self.knowledge_dict[predicate]:
                            i += 1
                            knowledge_value = self.knowledge_dict[predicate][tuple_key]
                            knowledge_value_neg = utils.knowledge_value_neg(knowledge_value)

                    if predicate_neg in self.knowledge_dict:
                        match = re.findall(r"\((.*?)\)", knowledge)
                        try:
                            assert len(match) == 1
                        except:
                            error_info = f"\n!!! Type-2 Error in Stage-2/Stage-3: Abstraction/Generation result format error by the perception LLM (line-282: generate_rules). \n\tThe reason may be there exist brackets in the arguments for the knowledge {knowledge}.\n\nFailure Type = 2, Failure Stage = 2"
                            self.error_info = error_info
                            print(error_info)
                            return 1

                        tuple_key = tuple([mat.strip() for mat in match[0].split(',')])
                        if tuple_key in self.knowledge_dict[predicate_neg]:
                            i += 1
                            knowledge_value_neg = self.knowledge_dict[predicate_neg][tuple_key]
                            knowledge_value = utils.knowledge_value_neg(knowledge_value_neg)

                    if i == 2:
                        try:
                            assert utils.knowledge_value_neg(knowledge_value) == knowledge_value_neg
                        except:
                            error_info = f"\n!!! Type-2 Error in Stage-2: Abstraction result format error by the perception LLM (line-296: generate_rules). \n\tWrong interpretation value for predicates {predicate} and {predicate_neg}. Only Unknown, True, False are allowed. Given knowledge value: {knowledge_value} and knowledge_value_neg {knowledge_value_neg}\n\nFailure Type = 2, Failure Stage = 2"
                            self.error_info = error_info
                            print(error_info)
                            return 1


                if len(free_var) > 0:
                    try:
                        assert knowledge_value == 'Unknown'
                    except:
                        error_info = f"\n!!! Type-3 Error in Stage-3 (line-305: generate_rules). The knowledge value for predicate with free vars should be Unknown. Here is {knowledge}={knowledge_value}.\n\nFailure Type = 9, Failure Stage = 3"
                        self.error_info = error_info
                        print(error_info)
                        return 1

                if knowledge in self.knowledge_prop_mapping:
                    prop_rule.append(self.knowledge_prop_mapping[knowledge])
                    prop_id = self.knowledge_prop_mapping[knowledge]
                    try:
                        assert prop_id in self.prop_dict
                        assert self.prop_dict[prop_id] == knowledge_value_enum[str(knowledge_value)]
                    except:
                        error_info = f"\n!!! Type-3 Error in Stage-3 (line-316: generate_rules). The knowledge value should not be conflicted to the prop_value.\n\nFailure Type = 9, Failure Stage = 3"
                        self.error_info = error_info
                        print(error_info)
                        return 1
                else:
                    prop_id = 'p_' + str(len(self.knowledge_prop_mapping))
                    if if_neg:
                        prop_id_neg = prop_id + '_neg'
                        self.knowledge_prop_mapping[knowledge] = prop_id_neg
                        prop_rule.append(prop_id_neg)
                        try:
                            assert prop_id_neg not in self.prop_dict
                        except:
                            error_info = f"\n!!! Type-3 Error in Stage-3 (line-330: generate_rules). The consistency of 'knowledge -> prop_id' mapping.\n\nFailure Type = 9, Failure Stage = 3"
                            self.error_info = error_info
                            print(error_info)
                            return 1

                        prop_neg_value = knowledge_value_enum[str(knowledge_value)]
                        self.prop_dict[prop_id_neg] = prop_neg_value
                        self.graph.add_node(prop_id_neg, type='lit', label=knowledge, free_var=list(free_var),
                                            value=prop_neg_value)

                    else:
                        self.knowledge_prop_mapping[knowledge] = prop_id
                        prop_rule.append(prop_id)
                        try:
                            assert prop_id not in self.prop_dict
                        except:
                            error_info = f"\n!!! Type-3 Error in Stage-3 (line-351: generate_rules). The consistency of 'knowledge -> prop_id' mapping.\n\nFailure Type = 9, Failure Stage = 3"
                            self.error_info = error_info
                            print(error_info)
                            return 1

                        prop_value = knowledge_value_enum[str(knowledge_value)]
                        self.prop_dict[prop_id] = prop_value
                        self.graph.add_node(prop_id, type='lit', label=knowledge, free_var=list(free_var),
                                            value=prop_value)  # initially set it as unknown

            prop_rule_list.append(prop_rule)
            return prop_rule_list

        current_var = unique_vars[var_index]

        if is_number(current_var):
            var_mapping[current_var] = remove_last_zero(current_var)
            prop_rule_list = self.generate_rules(var_index + 1, unique_vars, var_mapping, prop_rule_list, raw_rule, if_prop)
            return prop_rule_list

        clean_var = re.findall(r'\b[a-zA-Z_]\w*\b', current_var) # not include operators

        if len(clean_var)>1:
            var_mapping[current_var] = None
            prop_rule_list = self.generate_rules(var_index + 1, unique_vars, var_mapping, prop_rule_list, raw_rule, if_prop)
            return prop_rule_list

        possible_values = []
        test_possible_values = []
        for predicate, vars_tuple in raw_rule:
            # if current_var in vars_tuple and predicate in self.knowledge_dict:
            clean_predicate = predicate[1:] if predicate.startswith('~') else predicate
            # clean_predicate_neg = '~'+clean_predicate
            if current_var in vars_tuple and clean_predicate in self.raw_knowledge_dict:
                for key_tuple, value in self.raw_knowledge_dict[clean_predicate].items():
                    if len(key_tuple) != len(vars_tuple):
                        continue
                    bindings = {}
                    not_match = False
                    for p, v in zip(vars_tuple, key_tuple):
                        if is_number(p):  # it's a variable like 'x' or 'z'
                            try:
                                assert is_number(v)
                                assert remove_last_zero(str(float(p)))==remove_last_zero(str(v))
                            except:
                                not_match = True
                                break
                        else:  # it's a constant, must match exactly
                            bindings[p]=v
                    if current_var in bindings and (not not_match):
                        test_possible_values.append(bindings[current_var])
                    possible_values.append(key_tuple[vars_tuple.index(current_var)])

        possible_values = list(set(test_possible_values))
        if len(possible_values) == 0:
            var_mapping[current_var] = None
            prop_rule_list = self.generate_rules(var_index + 1, unique_vars, var_mapping, prop_rule_list, raw_rule, if_prop)
        else:
            for value in possible_values:
                var_mapping[current_var] = value
                prop_rule_list = self.generate_rules(var_index + 1, unique_vars, var_mapping, prop_rule_list, raw_rule, if_prop)

        return prop_rule_list


    def propositionalize_rule(self, raw_rule, if_prop):
        """
        Convert a predicate logic rule into propositional rules based on knowledge
        each rule has a unique value matrix, head is the unique vars
        :param raw_rule: the predicate logic rule (without operators) in the form of a dict of predicate (key: pred_name, value: argus)
        :return: a set of propositionalized rules
        """

        unique_vars = list(dict.fromkeys([var for _, var in raw_rule for var in var]).keys())
        prop_rule_list = self.generate_rules(0, unique_vars, {}, [], raw_rule, if_prop)

        return prop_rule_list


    def fulfill_var_mapping(self, var_mapping, new_var_mapping, raw_rule):
        unique_var_with_preserved = set()
        unique_var = set()
        combined_var_dict = {}
        free_var_set = set([key for key in var_mapping if var_mapping[key] is None])

        for key in var_mapping:
            clean_var = re.findall(r'\b[a-zA-Z_]\w*\b', key)
            if len(clean_var) == 1:
                unique_var.add(clean_var[0])
                unique_var_with_preserved.add(clean_var[0])
            elif len(clean_var) == 0:
                unique_var_with_preserved.add(key)
            else:
                combined_var_dict[key] = clean_var

        value_string_list = []
        raw_argus_string_list = []
        boolean_value_list = []
        for predicate, vars_tuple in raw_rule[:-1]:

            if not set(vars_tuple).isdisjoint(free_var_set):
                argus = [new_var_mapping[var] if new_var_mapping[var] is not None else var for var in vars_tuple]
            else:
                continue

            boolean_value = not predicate[0] == '~'
            if not boolean_value:
                predicate = predicate[1:]
            value_string = f"{predicate}({', '.join(argus)}) = {boolean_value}"
            value_string_list.append(value_string)
            raw_argus_string_list.append(vars_tuple)
            boolean_value_list.append(boolean_value)

        if len(value_string_list) >= 1:
            full_prompt = self.prompt_temp_config['newInstanceGen_prompt'].replace("[[Variables]]", f"{', '.join(self.variables)}")
            clean_values = [f"{pre.split('_')[0]}({pre.split('(')[1]}" for pre in value_string_list]
            full_prompt = full_prompt.replace("[[Predicates]]", '\n\t'.join(self.predicates))
            full_prompt = full_prompt.replace("[[Values]]", '\n\t'.join(clean_values))

            res, status = self.perception_model.generate_response(full_prompt=full_prompt, temp=0.5)

            try:
                assert status == True
            except:
                self.error_info = f"\n!!! APIError: {res}.\n\nFailure Type = 8, Failure Stage = 3"
                print(self.error_info)
                return 1

            invoke_log = f"\n@@@@@@@@@ We invoke oracle agent to find a new instance"
            if self.if_print:
                print(invoke_log)
                self.print_file.write(invoke_log)

            #\w match A-Z, a-z, 0-9, _, but not special characters like - . space
            if self.domain == 'comp' or self.domain == 'ineq':
                matches = re.findall(r'(\w+)\s*=\s*(-?\d+(?:\.\d+)?)', res)
            else: # match everything except a new line
                matches = re.findall(r'(\w+)\s*=\s*(.+)', res)

            instance_dict = {key: value for key, value in matches}

            if self.domain == 'comp' or self.domain == 'ineq':
                instance_dict = {key: remove_last_zero(value) for key, value in instance_dict.items()}

            for var in instance_dict:
                if var in new_var_mapping and new_var_mapping[var] is None:
                    new_var_mapping[var] = instance_dict[var]

            # update combined_argus
            for combined_argus in combined_var_dict:
                new_argus = set(combined_var_dict[combined_argus])
                value = combined_argus

                for sub_var in new_argus:
                    try:
                        assert new_var_mapping[sub_var] is not None
                    except:
                        self.error_info = f"\n!!! Type-2 Error in Stage-3: Perception LLM fails to generate the instance for the free variables in the given format (line-509: fulfill_var_mapping). Expected output: {{ ..., {sub_var} = ? ,...}}. Returned output: {res}\n\nFailure Type = 3, Failure Stage = 3"
                        print(self.error_info)
                        return 1

                    value = re.sub(rf'\b{sub_var}\b', new_var_mapping[sub_var], value)

                try:
                    if self.domain == 'comp' or self.domain == 'ineq':
                        eval_res = remove_last_zero(str(decimal_eval(value)))
                    else:
                        eval_res = new_argus
                except:
                    # TODO: Rational_text_based_math_value
                    eval_res = value.strip()

                new_var_mapping[combined_argus] = eval_res

            for index in range(len(value_string_list)):
                key = value_string_list[index].split('(')[0]
                value_key = tuple(new_var_mapping[raw_argus] for raw_argus in raw_argus_string_list[index])
                if key in self.knowledge_dict and value_key in self.knowledge_dict[key]:
                    try:
                        assert str(boolean_value_list[index]) == self.knowledge_dict[key][value_key]
                    except:
                        error_info = f"\n!!! Type-3 Error in Stage-3 (line-526: fulfill_var_mapping). Conflict knowledge value.\n\nFailure Type = 9, Failure Stage = 3"
                        self.error_info = error_info
                        print(error_info)
                        return 1
                else:
                    if key not in self.knowledge_dict:
                        self.knowledge_dict[key] = {}
                    self.knowledge_dict[key][value_key] = str(boolean_value_list[index])
                    instance_log = f"\n\tWe generate an instance: {key}({value_key})={boolean_value_list[index]}"
                    if self.if_print:
                        print(instance_log)
                        self.print_file.write(instance_log)

                key_neg = '~'+key
                if key_neg in self.knowledge_dict and value_key in self.knowledge_dict[key_neg]:
                    try:
                        assert str(not(boolean_value_list[index])) == self.knowledge_dict[key_neg][value_key]
                    except:
                        error_info = f"\n!!! Type-3 Error in Stage-3 (line-541: fulfill_var_mapping). Conflict knowledge value. \n\nFailure Type = 9, Failure Stage = 3"
                        self.error_info = error_info
                        print(error_info)
                        return 1
                else:
                    if key_neg not in self.knowledge_dict:
                        self.knowledge_dict[key_neg] = {}

                    self.knowledge_dict[key_neg][value_key] = str(not boolean_value_list[index])
                    instance_log = f"\n\tWe generate an instance: {key_neg}({value_key})={not boolean_value_list[index]}"
                    if self.if_print:
                        print(instance_log)
                        self.print_file.write(instance_log)

        return 0


    def graph_print(self, opt=0, out_path='./graph.json'):
        """
        print the computation graph with input options
        :param opt:
        0: just output nodes with nothing;
        1: output nodes and edges with data_original;
        2: print adjacency list;
        3: print edge list;
        4: export as JSON
        :return: None
        """
        if opt == 1:
            print("Nodes with attributes:", self.graph.nodes(data=True))
            print("Edges with attributes:", self.graph.edges(data=True))
        elif opt == 2:
            for node, adjacents in self.graph.adjacency():
                print(node, "->", dict(adjacents))
        elif opt == 3:
            for line in nx.generate_edgelist(self.graph, data=True):
                print(line)
        elif opt == 4:
            graph_data = nx.node_link_data(self.graph)
            with open(out_path, "w") as f:
                json.dump(graph_data, f, indent=4)
        else:  # other opt>=1 or option!= 0
            print("Nodes:", self.graph.nodes())
            print("Edges:", self.graph.edges())


    def graph_init_without_unk(self, print_opt=0):

        for rule in self.prop_rule_set:
            conditions = rule.replace(" ", '').split('=>')[0].split('&')
            conclusion = rule.replace(" ", '').split('=>')[1]
            LHS_node = rule.split('=>')[0].strip()  # p1 & p2
            if len(conditions)==1:
                single_cond = conditions[0]
                LHS_node = f"{single_cond} & {single_cond}"
                self.graph.add_node(LHS_node, type='lhs', value=2)  # unknown value is 2
            else:
                self.graph.add_node(LHS_node, type='lhs', value=2)  # unknown value is 2

            self.graph.add_edge(LHS_node, conclusion)
            for lit in conditions:
                self.graph.add_node(lit, type='lit', value=self.prop_dict[lit])  # update node values
                self.graph.add_edge(lit, LHS_node)

            self.graph.add_node(conclusion, type='lit', value=self.prop_dict[conclusion])

        if self.if_print or True:
            self.graph_print(opt=4)

        return 0


    def forward_chain(self, opt=0):
        """
        Performs forward chaining to derive new facts and detect inconsistencies: only based on the initial benchmark
        i) each LHS node represents a rule
        ii) opt=0, early stop once find an inconsistency; otherwise, find all inconsistencies.
        :return True->Consistent, False->Inconsistent
        """
        if_consistent = True
        for prop in self.proposition_set:
            if self.prop_dict[prop] == 0:
                print("\n\tWe find an inconsistency under the proposition rule: True =>", prop)
                pro_knowledge = next(
                    (k for k, v in self.knowledge_prop_mapping.items() if v == prop), None)
                print(f"The corresponding inconsistent knowledge is: {pro_knowledge} = False")
                if self.if_print:
                    self.print_file.write(f"\n\tWe find an inconsistency under the proposition rule: True => {prop}")
                    self.print_file.write(
                        f"\n\tThe corresponding inconsistent knowledge is: {pro_knowledge} = False")

                if_consistent = False

                if opt == 0:
                    break

        unchecked_LHS = [node for node, data in self.graph.nodes(data=True) if (
                data.get("type") == 'lhs' and data.get("value") == 2)]

        if self.if_print:
            print("unchecked_LHS list:", unchecked_LHS, '\n')

        inconsistent_LHS_set = []


        while len(unchecked_LHS) > 0:
            if not if_consistent and opt == 0:
                break
            LHS_node = unchecked_LHS[0]
            unchecked_LHS.remove(LHS_node)

            lhs_set = list(self.graph.predecessors(LHS_node))
            lhs_set_value = [self.graph.nodes[node].get("value") for node in lhs_set]
            if 2 in lhs_set_value:
                continue

            lhs_value_sum = int(sum(lhs_set_value))

            if lhs_value_sum == len(lhs_set_value):
                self.graph.nodes[LHS_node]["value"] = 1
                rhs_list = list(self.graph.successors(LHS_node))
                for rhs in rhs_list:
                    rhs_value = self.graph.nodes[rhs].get("value")
                    if rhs_value == 2:  # inferred new knowledge
                        self.graph.nodes[rhs]["value"] = 1
                        self.new_inferred_node.add(rhs)
                        self.new_inferred_node_rule_dict[rhs] = self.graph.predecessors(rhs)
                        print(f"We find a newly inferred knowledge: {self.graph.nodes[rhs]['label']} = True.")
                        if self.if_print:
                            self.print_file.write(f"\n\tWe find a newly inferred knowledge: {self.graph.nodes[rhs]['label']} = True.")
                    elif rhs_value == 1:
                        continue
                    else: # find an inconsistency
                        inconsistent_LHS_set.append(LHS_node)
                        if_consistent = False
                        print("\n\tWe find an inconsistency under the proposition rule:", lhs_set, '->', rhs)
                        if self.if_print:
                            self.print_file.write(f"\n\tWe find an inconsistency under the proposition rule: {lhs_set} -> {rhs}")
                        all_prop_set = lhs_set.copy()
                        all_prop_set.append(rhs)
                        all_knowledge_set = []
                        all_boolean_value_set = []
                        for lit in all_prop_set:
                            lit_pos_knowledge = next(
                                (k for k, v in self.knowledge_prop_mapping.items() if v == lit), None)
                            all_knowledge_set.append(lit_pos_knowledge)
                            all_boolean_value_set.append(str(self.prop_dict[lit]))

                        knowledge_rule = (' & ').join(all_knowledge_set[:-1]) + ' => ' + all_knowledge_set[-1]
                        value_rule = (' & ').join(all_boolean_value_set[:-1]) + ' => ' + all_boolean_value_set[-1]
                        print(f"\tThe corresponding inconsistent knowledge is: {knowledge_rule}")
                        print(f"\tThe corresponding Boolean value mapping is: {value_rule} (0: False, 1: True)")
                        if self.if_print:
                            self.print_file.write(f"\n\tThe corresponding inconsistent knowledge is: {knowledge_rule}")
                            self.print_file.write(f"\n\tThe corresponding Boolean value mapping is: {value_rule} (0: False, 1: True)")

                        if opt == 0:
                            break

            else:
                self.graph.nodes[LHS_node]["value"] = 0

        if if_consistent:
            print("We check all the rule given in the specification ESL.file and no inconsistencies are found in the context.")
            return True
        else:
            return False

    def update_graph(self, new_props):
        return

    def delete(self):
        self.graph.clear()
        self.variables.clear()
        self.predicates.clear()
        self.rule_set.clear()
        self.knowledge_dict.clear()
        self.knowledge_prop_mapping.clear()
        self.prop_dict.clear()
        self.prop_rule_set.clear()