import numpy as np
from openai import OpenAI, OpenAIError
import openai
from google.genai import types

import dashscope
from google import genai
from google.genai import errors

class TargetAgent:
    def __init__(self, api_config, if_memory, timeout):
        self.model_name = api_config['target_model_name']
        self.api_key = api_config['target_api_key']
        self.api_base_url = api_config['target_base_url']
        self.max_tokens = api_config['target_max_tokens']
        self.if_memory = if_memory
        self.google_chat = None
        self.timeout = timeout
        if 'gemini' in self.model_name:
            if self.if_memory>0:
                self.llm_agent = genai.Client(api_key=self.api_key).chats.create(model=self.model_name)
            else:
                self.llm_agent = genai.Client(api_key=self.api_key).models
        else:
            self.llm_agent = OpenAI(
                api_key=self.api_key,
                base_url=self.api_base_url
            )
        self.chat_history = []

    def generate_response(self, full_prompt, config=None, temp=0, top_p=0.2):

        def openai_response(full_prompt,temp,top_p):
            if self.model_name == 'o1-mini':
                if self.if_memory > 0:
                    self.chat_history.append({'role': 'user', 'content': full_prompt})
                else:
                    self.chat_history = [{'role': 'user', 'content': full_prompt}]

                completion = self.llm_agent.chat.completions.create(
                    model=self.model_name,
                    messages=self.chat_history,
                    timeout=self.timeout,
                )
            else:
                if self.if_memory > 0:
                    self.chat_history.append({'role': 'user', 'content': full_prompt})
                else:
                    self.chat_history = [{'role': 'user', 'content': full_prompt}]

                completion = self.llm_agent.chat.completions.create(
                    model=self.model_name,
                    messages=self.chat_history,
                    temperature=temp,
                    top_p=top_p,
                    max_tokens=self.max_tokens,
                    timeout=self.timeout,
                )
            return completion.choices[0].message.content

        def google_response(full_prompt,temp,top_p):
            config = types.GenerateContentConfig(
                max_output_tokens=self.max_tokens,
                temperature=temp
            )

            if self.if_memory > 0:
                response = self.llm_agent.send_message(full_prompt, config=config)
            else:
                response = self.llm_agent.generate_content(model=self.model_name, contents=full_prompt, config=config)

            return response.text.strip()

        if 'gemini' in self.model_name:
            try:
                return google_response(full_prompt,temp,top_p), True
            except errors.APIError as e:
                print(e)
                print(e.message)
                return e.message, False
            except errors.ClientError as e:
                print(e)
                print(e.message)
                return e.message, False
        else:
            try:
                return openai_response(full_prompt, temp, top_p), True
            except OpenAIError as e:
                print(e)
                return e, False


class PerceptionAgent:
    def __init__(self, api_config, timeout):
        self.model_name = api_config['perception_model_name']
        self.api_key = api_config['perception_api_key']
        self.api_base_url = api_config['perception_base_url']
        self.max_tokens = api_config['perception_max_tokens']
        self.timeout = timeout
        self.google_chat = None
        if 'gemini' in self.model_name:
            self.llm_agent = genai.Client(api_key=self.api_key).models
        else:
            self.llm_agent = OpenAI(
                api_key=self.api_key,
                base_url=self.api_base_url
            )
        self.chat_history = []


    def generate_response(self, full_prompt='', config=None, temp=0, top_p=0.2):

        def openai_response(full_prompt, temp, top_p):

            if self.model_name == 'o1-mini':
                completion = self.llm_agent.chat.completions.create(
                    model=self.model_name,
                    messages=[
                        {'role': 'user', 'content': full_prompt}
                    ],
                    timeout=self.timeout,
                )
            else:
                completion = self.llm_agent.chat.completions.create(
                    model=self.model_name,
                    messages=[
                        {'role': 'user', 'content': full_prompt}
                    ],
                    temperature=temp,
                    top_p=top_p,
                    max_tokens=self.max_tokens,
                    timeout=self.timeout,
                )
            return completion.choices[0].message.content

        def google_response(full_prompt, temp, top_p):
            config = types.GenerateContentConfig(
                max_output_tokens=self.max_tokens,
                temperature=temp
            )
            response = self.llm_agent.generate_content(model=self.model_name, contents=full_prompt, config=config)

            return response.text.strip()

        if 'gemini' in self.model_name:
            try:
                return google_response(full_prompt, temp, top_p), True
            except errors.APIError as e:
                print(e)
                print(e.message)
                return e.message, False
            except errors.ClientError as e:
                print(e)
                print(e.message)
                return e.message, False
        else:
            try:
                return openai_response(full_prompt, temp, top_p), True
            except OpenAIError as e:
                print(e)
                return e, False