import numpy as np
import random as rn
from numpy.random import default_rng
from data_engineering.fbm import FBM
from scipy.stats import expon, norm
import matplotlib.pyplot as plt

SEED = 42
np.random.seed(SEED)
rn.seed(SEED)
EPS = 1e-8

def generate_brownian_asset_dynamics(option_parameters_dict, N, rs=SEED):

    r = option_parameters_dict['risk_free_rate']
    delta = option_parameters_dict['dividend_yield']
    sigma = option_parameters_dict['volatility_sigma']
    T = option_parameters_dict['time_horizon_yrs']
    L = option_parameters_dict['num_exercise_opportunities'] - 1
    d = option_parameters_dict['num_assets']
    s0 = option_parameters_dict['current_price']
    K = option_parameters_dict['option_strike_price']
    rhoij = option_parameters_dict['rhoij']
    option_type = option_parameters_dict['option_type']


    np.random.seed(rs)

    X = np.zeros((N, L+1, d))

    rho = np.identity(d)
    rho[np.nonzero(1 - rho)] = rhoij
    mu = r - delta
    dt = T / L

    # correlated multi-dimensional Black-Scholes
    C = np.linalg.cholesky(rho)
    St = (mu - sigma ** 2 / 2) * dt + sigma * np.transpose(np.dot(C, np.random.normal(0, np.sqrt(dt), size=(L, d, N))), (1, 2, 0))
    St = s0 * np.exp(St.cumsum(axis=0))
    St = np.concatenate((s0 * np.ones((1, N, d)), St), axis=0)
    X = np.transpose(St,(1,0,2))

    time = np.linspace(0, T, L + 1)
    tt = np.full(shape=(N, L + 1), fill_value=time)
    if option_type == 'american_geometric_call':
        R = np.clip((np.prod(X**(1.0/d), axis=2)) - K, 0, None)
    elif option_type == 'bermudan_max_call':
        R = np.clip(np.amax(X, axis=2) - K, 0, None) * np.exp(-r * tt)
    else:
        print('unsupported option type')
        R = None

    return X, R


def generate_fractional_motion(fbm_parameters_dict, N, rs=SEED):
    L = fbm_parameters_dict['num_steps'] - 1
    hurst = fbm_parameters_dict['hurst']
    length = fbm_parameters_dict['length']
    method = fbm_parameters_dict['method']
    f = FBM(L-1, hurst, length, method)

    np.random.seed(rs)

    X = np.zeros((N, L+1 , 1 + 2))

    for i in range(N):
        X[i, 1:, 0] = f.fbm()
    X[:, :, -1] = np.linspace(0, L+1, num=L+1)/(L+1)
    X[:, :, -2] = 1-np.linspace(0, L+1, num=L+1)/(L+1)


    r_min = np.amin(X[:, :, 0].ravel())
    print(r_min)
    R = X[:, :, 0] - r_min

    return X, R, r_min



