import numpy as np
import tensorflow as tf
print(tf.__version__)
import random as rn

import itertools
import pandas as pd

from data_engineering.analyze_data import analyze
from data_engineering.generate_synthetic_dataset import generate_fractional_motion
from models.dos_model import train_dos_model
from models.fqi_model import train_fqi_model
from models.ospg_model import train_ospg_model
from models.rrlsm_model import train_rrlsm_model


#REPRODUCIBILITY
SEED = 42
import os
import time
os.environ['PYTHONHASHSEED'] = '0'
os.environ['CUDA_VISIBLE_DEVICES'] = '0'
np.random.seed(SEED)
rn.seed(SEED)
tf.random.set_seed(SEED)

TRAIN_FRACTION = 0.5
VAL_FRACTION = 0.2

RESULTS_DIR = './'
EXPERIMENT_FNAME = RESULTS_DIR + 'fbm_experiment.csv'

FOLDS = 10
HURST = [0.05, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 0.95]
NUM_TRAJECTORIES = [40000]


def main():
    start_time = time.time()
    experiment_df = pd.DataFrame(
    {
        'algorithm': [],
        'hurst': [],
        'num_trajectories': [],
        'fold': [],
        'reward': [],
        'train_time': [],
        'prediction_time_per_ts': []
    })


    experiments = list(itertools.product(HURST, NUM_TRAJECTORIES))
    exp_ctr = 0
    exp_row = 0
    for experiment_params in experiments:
        exp_ctr = exp_ctr + 1
        fbm_parameters = {
            'num_steps': 101,
            'hurst': experiment_params[0],
            'length': 1,
            'method': 'cholesky'
        }

        X, R, r_offset = generate_fractional_motion(fbm_parameters, experiment_params[1], rs=42)
        data_stats_dict = analyze(X, R, TRAIN_FRACTION, VAL_FRACTION, FOLDS)

        ospg_config = {
            'batch_size': 64,
            'os_epochs': 100,
            'samples_per_epoch': 200,
            'os_lr': 0.001,
            'clipnorm': 5,
            'use_DNN': False,
            'include_R': True,
            'num_stacked_layers': 1,
            'units_hidden': 20
        }

        rrlsm_config = {
            'samples_per_epoch': 200,
            'num_stacked_layers': 1,
            'ker_std': 0.0001,
            'rec_std': 0.3,
            'include_R': True,
            'units_hidden': 20
        }

        dos_config = {
            'batch_size': 64,
            'dos_epochs': 100,
            'samples_per_epoch': 200,
            'dos_lr': 0.001,
            'clipnorm': 5,
            'omit_time_zero': True,
            'make_markovian': True,
            'num_stacked_layers': 2,
            'units_hidden': 20
        }

        fqi_config = {
            'batch_size': 64,
            'q_epochs': 100,
            'samples_per_epoch': 200,
            'q_lr': 0.001,
            'clipnorm': 5,
            'use_DNN': False,
            'include_R': True,
            'num_stacked_layers': 1,
            'units_hidden': 20
        }

        ## DOS
        dos_result = train_dos_model(dos_config, data_stats_dict, transform_str=None, make_markovian=dos_config['make_markovian'])
        for fold in range(FOLDS):
            experiment_df.loc[exp_row] = ['DOS_ES', fbm_parameters['hurst'],
                                          experiment_params[1],
                                          fold, dos_result['dos_rewards'][fold]+r_offset,
                                          dos_result['train_times'][fold],
                                          dos_result['prediction_time_per_ts'][fold]]
            exp_row = exp_row + 1

        ## RRLSM
        rrlsm_result = train_rrlsm_model(rrlsm_config, data_stats_dict, transform_str=None)
        for fold in range(FOLDS):
            experiment_df.loc[exp_row] = ['RNN_RRLSM', fbm_parameters['hurst'],
                                          experiment_params[1],
                                          fold, rrlsm_result['rrlsm_rewards'][fold]+r_offset,
                                          rrlsm_result['train_times'][fold],
                                          rrlsm_result['prediction_time_per_ts'][fold]]
            exp_row = exp_row + 1
            print(str(rrlsm_result['rrlsm_rewards'][fold]+r_offset))

        ## FQI
        q_result = train_fqi_model(fqi_config, data_stats_dict, transform_str=None)
        for fold in range(FOLDS):
            experiment_df.loc[exp_row] = ['RNN_FQI', fbm_parameters['hurst'],
                                          experiment_params[1],
                                          fold, q_result['q_rewards'][fold]+r_offset,
                                          q_result['train_times'][fold],
                                          q_result['prediction_time_per_ts'][fold]]
            exp_row = exp_row + 1
            print(str(q_result['q_rewards'][fold] + r_offset))

        ## OSPG
        os_result = train_ospg_model(ospg_config, data_stats_dict, transform_str=None)
        for fold in range(FOLDS):
            experiment_df.loc[exp_row] = ['RNN_OSPG', fbm_parameters['hurst'],
                                          experiment_params[1],
                                          fold, os_result['os_rewards'][fold]+r_offset,
                                          os_result['train_times'][fold],
                                          os_result['prediction_time_per_ts'][fold]]
            exp_row = exp_row + 1
            print(str(os_result['os_rewards'][fold] + r_offset))

        print('done_experiment %d of %d' % (exp_ctr, len(experiments)))

    experiment_df.to_csv(path_or_buf=EXPERIMENT_FNAME, index=False)
    end_time=time.time()
    print('run_time(min): ' + str((end_time-start_time)/60))
    print("done")


if __name__ == '__main__':
    main()