import torch
import numpy as np

from utils.utils import torch_rand


def apply_rand_phase(signal):
    phase = 2.0 * np.pi * torch_rand()
    return signal * np.exp(phase * 1j)


def apply_doppler(signal, freq):
    phases = torch.arange(signal.shape[0]) / freq
    return signal * torch.exp(1j * phases)


def apply_shadow_fading(signal, sine_db, sine_freq, noise_db):
    n = signal.shape[0]
    smooth = torch.sin(2 * np.pi * (torch.arange(n) / sine_freq + torch_rand())) * sine_db
    noise = torch.randn((n,)) * noise_db
    return signal * 10 ** ((smooth + noise) / 20)


def generate_freq(lo, hi):
    t = torch_rand()
    return int(round(hi ** t * lo ** (1 - t)))


def apply_transforms(signal, transforms):
    for transform in transforms:
        name = transform["name"]
        if name == "rotate":
            signal = apply_rand_phase(signal)
        elif name == "doppler":
            freq_lo = transform["freq_lo"]
            freq_hi = transform["freq_hi"]
            freq = generate_freq(freq_lo, freq_hi)
            signal = apply_doppler(signal, freq)
        elif name == "shadow_fading":
            freq_lo = transform["freq_lo"]
            freq_hi = transform["freq_hi"]
            sine_freq = generate_freq(freq_lo, freq_hi)
            sine_db = torch_rand() * transform["max_sine_db"]
            noise_db = torch_rand() * transform["max_noise_db"]
            signal = apply_shadow_fading(signal, sine_db, sine_freq, noise_db)
        else:
            raise ValueError(f"Unknown transform name: {name}")
    return signal
            
