import inspect
import os
from typing import Any, Callable, List, Mapping
from ml_collections import ConfigDict


def get_default_args(func):
    signature = inspect.signature(func)
    return {
        k: v.default
        for k, v in signature.parameters.items()
        if v.default is not inspect.Parameter.empty
    }


# We expand env var in strings, following lists and dicts recursively
# Can also support tuples in the future, if needed
def expand_os_env_in_confg(config):
    if isinstance(config, dict) or isinstance(config, ConfigDict):
        for k, v in config.items():
            if isinstance(v, str):
                config[k] = os.path.expandvars(v)
            elif isinstance(v, dict) or isinstance(v, list) or isinstance(v, ConfigDict):
                expand_os_env_in_confg(v)
    elif isinstance(config, list):
        for i in range(len(config)):
            if isinstance(config[i], str):
                config[i] = os.path.expandvars(config[i])
            elif isinstance(config[i], dict) or isinstance(config[i], list) or isinstance(config[i], ConfigDict):
                expand_os_env_in_confg(config[i])


class ClassBuilder:
    def __init__(self, register: Mapping[str, Callable]):
        self.register = register

    def is_registered(self, config: List[Any]) -> bool:
        if config[0] not in self.register:
            return False
        return True

    def build_class(self, config: List[Any], **kwargs):
        if not self.is_registered(config):
            raise ValueError(f"{config[0]} has not been registered!")
        expand_os_env_in_confg(kwargs)
        if len(config) == 1:
            return self.register[config[0]](**kwargs)
        else:
            expand_os_env_in_confg(config[1])
            return self.register[config[0]](**{**config[1], **kwargs})

    def build_dataclass(self, config: List[Any], **kwargs):
        if not self.is_registered(config):
            raise ValueError(f"{config[0]} has not been registered!")
        expand_os_env_in_confg(kwargs)
        if len(config) == 1:
            return ConfigDict(
                {
                    **get_default_args(self.register[config[0]]), 
                    **kwargs
                }
            )
        else:
            expand_os_env_in_confg(config[1])
            return ConfigDict(
                {
                    **get_default_args(self.register[config[0]]), 
                    **config[1], 
                    **kwargs
                }
            )

    def build(self, config: List[Any], **kwargs):
        return self.build_class(config, **kwargs), self.build_dataclass(config)
