from metagen.util import format_float, round_to_1_sig_fig
from enum import Enum
from random import randint, sample, random
import itertools

class PropertyGenerality(Enum):
    OVERALL = 0
    DIRECTIONAL_1 = 1
    DIRECTIONAL_2 = 2
    DIRECTIONAL_3 = 3
    DIRECTIONAL_12 = 4
    DIRECTIONAL_21 = 5
    DIRECTIONAL_23 = 6
    DIRECTIONAL_32 = 7
    DIRECTIONAL_13 = 8
    DIRECTIONAL_31 = 9

class PropertyType(Enum):
    ANISOTROPY = 0
    YOUNGS_MOD = 1
    SHEAR_MOD = 2
    POISSON_RATIO = 3
    BULK_MOD = 4
    VOLUME_FRACTION = 5

class TargetType(Enum):
    VALUE = 0
    UPPER_BOUND = 1
    LOWER_BOUND = 2
    RANGE = 3
    CATEGORICAL = 4

idx2component = {1: {"axis":"x", "axis-orientation":"horizontal"}, 
                 2: {"axis":"y", "axis-orientation":"vertical"},
                 3: {"axis":"z", "axis-orientation":"horizontal"}}

#TODO: ideally, the dataset coverage would be inferred (we also don't need it for the train/test sets; 
# we only need if for sampling an inverse design profile not from the dataset)
property_references = {
    # ----- Anistropy Information -----
    'A': {
        "full_prop_name": "Universal Anisotropy Index",
        "alternate_descriptors": ["UAI"],
        "property_generality": PropertyGenerality.OVERALL,
        "property_type": PropertyType.ANISOTROPY,
        "dataset_coverage": {
            "min": 0,
            "max": 70000,
            "q1": 0,
            "q3": 5,
            "densely_populated_ranges": [[0, 300]]
        },
        "smallest_meaningful_quantization": 0.001,
        "adjective_descriptors":[{"description": "isotropic", "target_type": TargetType.VALUE, "target_value":0},
                                 {"description": "nearly isotropic", "target_type": TargetType.UPPER_BOUND, "target_value":0.05},
                                 {"description": "anisotropic", "target_type": TargetType.LOWER_BOUND, "target_value":0.05},
                                 {"description": "highly anisotropic", "target_type": TargetType.LOWER_BOUND, "target_value": 1.0}],
        "property_descriptors": [{"description": "identical behavior in every direction", "target_type": TargetType.VALUE, "target_value":0},
                                 {"description": "a low universal anisotropy index", "target_type": TargetType.UPPER_BOUND, "target_value":0.05},
                                 {"description": "a high universal anisotropy index", "target_type": TargetType.LOWER_BOUND, "target_value": 1.0}]
    },
    # ----- Young's Modulus Information -----
    'E': {
        "full_prop_name": "Young's modulus",
        "alternate_symbols": ["E_{VRH}", "E_{avg}", "Voigt-Reuss-Hill Young's Modulus"],
        "property_generality": PropertyGenerality.OVERALL,
        "property_type": PropertyType.YOUNGS_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 1,
            "q1": 0.01,
            "q3": 0.08,
            "densely_populated_ranges": [[0, 0.2], [0.3, 0.6]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": "stiff", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},
                                 {"description": "compliant", "target_type": TargetType.UPPER_BOUND, "target_value":0.4}],
        "property_descriptors": [{"description": "a very low Young's modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": "a low Young's modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.25},
                                 {"description": "a high Young's modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.4},
                                 {"description": "a very high Young's modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},]
    },
    "E_1": {
        "full_prop_name": "directional Young's modulus",
        "alternate_symbols": [f"E_{idx2component[1]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_1,
        "property_type": PropertyType.YOUNGS_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 1,
            "q1": 0.01,
            "q3": 0.09,
            "densely_populated_ranges": [[0, 0.2], [0.3, 0.6]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"stiff along the {idx2component[1]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},
                                 {"description": f"compliant along the {idx2component[1]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.4}],
        "property_descriptors": [{"description": f"a very low Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a low Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.25},
                                 {"description": f"a high Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.4},
                                 {"description": f"a very high Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},]
    },
    "E_2": {
        "full_prop_name": "directional Young's modulus",
        "alternate_symbols": [f"E_{idx2component[2]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_2,
        "property_type": PropertyType.YOUNGS_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 1,
            "q1": 0.01,
            "q3": 0.09,
            "densely_populated_ranges": [[0, 0.2], [0.3, 0.6]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"stiff along the {idx2component[2]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},
                                 {"description": f"compliant along the {idx2component[2]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.4}],
        "property_descriptors": [{"description": f"a very low Young's modulus along the {idx2component[2]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a very low Young's modulus along the {idx2component[2]['axis-orientation']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a low Young's modulus along the {idx2component[2]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.25},
                                 {"description": f"a low Young's modulus along the {idx2component[2]['axis-orientation']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.25},
                                 {"description": f"a high Young's modulus along the {idx2component[2]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.4},
                                 {"description": f"a high Young's modulus along the {idx2component[2]['axis-orientation']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.4},
                                 {"description": f"a very high Young's modulus along the {idx2component[2]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},
                                 {"description": f"a very high Young's modulus along the {idx2component[2]['axis-orientation']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},]
    },  
    "E_3": {
        "full_prop_name": "directional Young's modulus",
        "alternate_symbols": [f"E_{idx2component[3]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_3,
        "property_type": PropertyType.YOUNGS_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 1,
            "q1": 0.01,
            "q3": 0.09,
            "densely_populated_ranges": [[0, 0.2], [0.35, 0.6]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"stiff along the {idx2component[1]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},
                                 {"description": f"compliant along the {idx2component[1]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.4}],
        "property_descriptors": [{"description": f"a very low Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a low Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.UPPER_BOUND, "target_value":0.25},
                                 {"description": f"a high Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.4},
                                 {"description": f"a very high Young's modulus along the {idx2component[1]['axis']} direction", "target_type": TargetType.LOWER_BOUND, "target_value":0.6},]
    },  

    # ----- Shear Modulus Information -----
    'G': {
        "full_prop_name": "shear modulus",
        "alternate_symbols": ["G_{avg}", "G_{VRH}",  "Voigt-Reuss-Hill Shear Modulus"],
        "property_generality": PropertyGenerality.OVERALL,
        "property_type": PropertyType.SHEAR_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 0.35,
            "q1": 0.003,
            "q3": 0.03,
            "densely_populated_ranges": [[0, 0.07], [0.12, 0.2]]
        },
        "smallest_meaningful_quantization": 0.001,
        "adjective_descriptors":[{"description": f"fluid-like", "target_type": TargetType.UPPER_BOUND, "target_value":0.01},
                                 {"description": f"resistant to shear forces", "target_type": TargetType.LOWER_BOUND, "target_value":0.2},
                                 {"description": f"rigid", "target_type": TargetType.LOWER_BOUND, "target_value":0.2},
                                 {"description": f"very resistant to shear forces", "target_type": TargetType.LOWER_BOUND, "target_value":0.3},
                                 {"description": f"very rigid", "target_type": TargetType.LOWER_BOUND, "target_value":0.3}],
        "property_descriptors": [{"description": f"a very low shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.01},
                                 {"description": f"a low shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a high shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.2},
                                 {"description": f"a very high shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.3}]
    },
    "G_23": {
        "full_prop_name": "directional shear modulus",
        "alternate_symbols": [f"G_{idx2component[2]['axis']}{idx2component[3]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_23,
        "property_type": PropertyType.SHEAR_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 0.35,
            "q1": 0.003,
            "q3": 0.03,
            "densely_populated_ranges": [[0, 0.06], [0.125, 0.2]]
        },
        "smallest_meaningful_quantization": 0.001,
        "adjective_descriptors":[{"description": f"resistant to particular shear forces", "target_type": TargetType.LOWER_BOUND, "target_value":0.2}],
        "property_descriptors": [{"description": f"a very low directional shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.01},
                                 {"description": f"a low directional shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a high directional shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.2},
                                 {"description": f"a very high directional shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.3}]
    },  
    "G_31": {
        "full_prop_name": "directional shear modulus",
        "alternate_symbols": [f"G_{idx2component[3]['axis']}{idx2component[1]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_31,
        "property_type": PropertyType.SHEAR_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 0.35,
            "q1": 0.004,
            "q3": 0.03,
            "densely_populated_ranges": [[0, 0.06], [0.12, 0.2]]
        },
        "smallest_meaningful_quantization": 0.001,
        "adjective_descriptors":[{"description": f"resistant to particular shear forces", "target_type": TargetType.LOWER_BOUND, "target_value":0.2}],
        "property_descriptors": [{"description": f"a very low directional shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.01},
                                 {"description": f"a low directional shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a high directional shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.2},
                                 {"description": f"a very high directional shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.3}]
    },    
    "G_12": {
        "full_prop_name": "directional shear modulus",
        "alternate_symbols": [f"G_{idx2component[1]['axis']}{idx2component[2]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_12,
        "property_type": PropertyType.SHEAR_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 0.35,
            "q1": 0.003,
            "q3": 0.03,
            "densely_populated_ranges": [[0, 0.06], [0.12, 0.2]]
        },
        "smallest_meaningful_quantization": 0.001,
        "adjective_descriptors":[{"description": f"resistant to particular shear forces", "target_type": TargetType.LOWER_BOUND, "target_value":0.2}],
        "property_descriptors": [{"description": f"a very low directional shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.01},
                                 {"description": f"a low directional shear modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.1},
                                 {"description": f"a high directional shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.2},
                                 {"description": f"a very high directional shear modulus", "target_type": TargetType.LOWER_BOUND, "target_value":0.3}]
    },    

    # ----- Poisson Ratio Information -----
    'nu': {
        "full_prop_name": "Poisson ratio",
        "alternate_symbols": ["nu_{VRH}"],
        "property_generality": PropertyGenerality.OVERALL,
        "property_type": PropertyType.POISSON_RATIO,
        "dataset_coverage": {
            "min": -0.5,
            "max": 0.5,
            "q1": 0.3,
            "q3": 0.36,
            "densely_populated_ranges": [[0.2, 0.4]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"auxetic", "target_type": TargetType.UPPER_BOUND, "target_value":0}],
        "property_descriptors": [{"description": f"a negative Poisson ratio", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"a positive Poisson ratio", "target_type": TargetType.LOWER_BOUND, "target_value":0}],
        "verb_descriptors":     [{"description": f"contracts transversely under axial compression", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands transversely under axial compression", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"contracts in other directions when compressed along one axis", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands in other directions when compressed along one axis", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands transversely under axial elongation", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts transversely under axial elongation", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands in other directions when stretched along one axis", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts in other directions when stretched along one axis", "target_type": TargetType.LOWER_BOUND, "target_value":0}]
    },   
    "nu_12": {
        "full_prop_name": "directional Poisson ratio",
        "alternate_symbols": [f"nu_{idx2component[1]['axis']}{idx2component[2]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_12,
        "property_type": PropertyType.POISSON_RATIO,
        "dataset_coverage": {
            "min": -2.3,
            "max": 3.05,
            "q1": 0.27,
            "q3": 0.43,
            "densely_populated_ranges": [[-0.1, 1]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"directionally auxetic", "target_type": TargetType.UPPER_BOUND, "target_value":0}],
        "property_descriptors": [{"description": f"a negative Poisson ratio in at least one direction", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"a positive Poisson ratio in at least one direction", "target_type": TargetType.LOWER_BOUND, "target_value":0}],
        "verb_descriptors":     [{"description": f"contracts in the {idx2component[2]['axis']} direction when the {idx2component[1]['axis']} direction is compressed", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[2]['axis']} direction when the {idx2component[1]['axis']} direction is compressed", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[2]['axis']} direction when the {idx2component[1]['axis']} direction is stretched", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts in the {idx2component[2]['axis']} direction when the {idx2component[1]['axis']} direction is stretched", "target_type": TargetType.LOWER_BOUND, "target_value":0}]
    },   
    "nu_13": {
        "full_prop_name": "directional Poisson ratio",
        "alternate_symbols": [f"nu_{idx2component[1]['axis']}{idx2component[3]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_13,
        "property_type": PropertyType.POISSON_RATIO,
        "dataset_coverage": {
            "min": -1,
            "max": 2.5,
            "q1": 0.27,
            "q3": 0.44,
            "densely_populated_ranges": [[-0.05, 1]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"directionally auxetic", "target_type": TargetType.UPPER_BOUND, "target_value":0}],
        "property_descriptors": [{"description": f"a negative Poisson ratio in at least one direction", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"a positive Poisson ratio in at least one direction", "target_type": TargetType.LOWER_BOUND, "target_value":0}],
        "verb_descriptors":     [{"description": f"contracts in the {idx2component[3]['axis']} direction when the {idx2component[1]['axis']} direction is compressed", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[3]['axis']} direction when the {idx2component[1]['axis']} direction is compressed", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[3]['axis']} direction when the {idx2component[1]['axis']} direction is stretched", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts in the {idx2component[3]['axis']} direction when the {idx2component[1]['axis']} direction is stretched", "target_type": TargetType.LOWER_BOUND, "target_value":0}]
    }, 
    "nu_23": {
        "full_prop_name": "directional Poisson ratio",
        "alternate_symbols": [f"nu_{idx2component[2]['axis']}{idx2component[3]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_23,
        "property_type": PropertyType.POISSON_RATIO,
        "dataset_coverage": {
            "min": -1,
            "max": 2.33,
            "q1": 0.27,
            "q3": 0.42,
            "densely_populated_ranges": [[-0.05, 0.6]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"directionally auxetic", "target_type": TargetType.UPPER_BOUND, "target_value":0}],
        "property_descriptors": [{"description": f"a negative Poisson ratio in at least one direction", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"a positive Poisson ratio in at least one direction", "target_type": TargetType.LOWER_BOUND, "target_value":0}],
        "verb_descriptors":     [{"description": f"contracts in the {idx2component[3]['axis']} direction when the {idx2component[2]['axis']} direction is compressed", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[3]['axis']} direction when the {idx2component[2]['axis']} direction is compressed", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[3]['axis']} direction when the {idx2component[2]['axis']} direction is stretched", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts in the {idx2component[3]['axis']} direction when the {idx2component[2]['axis']} direction is stretched", "target_type": TargetType.LOWER_BOUND, "target_value":0}]
    }, 
    "nu_21": {
        "full_prop_name": "directional Poisson ratio",
        "alternate_symbols": [f"nu_{idx2component[2]['axis']}{idx2component[1]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_21,
        "property_type": PropertyType.POISSON_RATIO,
        "dataset_coverage": {
            "min": -1,
            "max": 2,
            "q1": 0.29,
            "q3": 0.44,
            "densely_populated_ranges": [[-0.05, 1]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"directionally auxetic", "target_type": TargetType.UPPER_BOUND, "target_value":0}],
        "property_descriptors": [{"description": f"a negative Poisson ratio in at least one direction", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"a positive Poisson ratio in at least one direction", "target_type": TargetType.LOWER_BOUND, "target_value":0}],
        "verb_descriptors":     [{"description": f"contracts in the {idx2component[1]['axis']} direction when the {idx2component[2]['axis']} direction is compressed", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[1]['axis']} direction when the {idx2component[2]['axis']} direction is compressed", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[1]['axis']} direction when the {idx2component[2]['axis']} direction is stretched", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts in the {idx2component[1]['axis']} direction when the {idx2component[2]['axis']} direction is stretched", "target_type": TargetType.LOWER_BOUND, "target_value":0}]
    }, 
    "nu_31": {
        "full_prop_name": "directional Poisson ratio",
        "alternate_symbols": [f"nu_{idx2component[3]['axis']}{idx2component[1]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_31,
        "property_type": PropertyType.POISSON_RATIO,
        "dataset_coverage": {
            "min": -1,
            "max": 2.33,
            "q1": 0.29,
            "q3": 0.44,
            "densely_populated_ranges": [[-0.05, 1]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"directionally auxetic", "target_type": TargetType.UPPER_BOUND, "target_value":0}],
        "property_descriptors": [{"description": f"a negative Poisson ratio in at least one direction", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"a positive Poisson ratio in at least one direction", "target_type": TargetType.LOWER_BOUND, "target_value":0}],
        "verb_descriptors":     [{"description": f"contracts in the {idx2component[1]['axis']} direction when the {idx2component[3]['axis']} direction is compressed", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[1]['axis']} direction when the {idx2component[3]['axis']} direction is compressed", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[1]['axis']} direction when the {idx2component[3]['axis']} direction is stretched", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts in the {idx2component[1]['axis']} direction when the {idx2component[3]['axis']} direction is stretched", "target_type": TargetType.LOWER_BOUND, "target_value":0}]
    }, 
    "nu_32": {
        "full_prop_name": "directional Poisson ratio",
        "alternate_symbols": [f"nu_{idx2component[3]['axis']}{idx2component[2]['axis']}"],
        "property_generality": PropertyGenerality.DIRECTIONAL_32,
        "property_type": PropertyType.POISSON_RATIO,
        "dataset_coverage": {
            "min": -2.28,
            "max": 3.04,
            "q1": 0.28,
            "q3": 0.43,
            "densely_populated_ranges": [[-0.05, 1]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": f"directionally auxetic", "target_type": TargetType.UPPER_BOUND, "target_value":0}],
        "property_descriptors": [{"description": f"a negative Poisson ratio in at least one direction", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"a positive Poisson ratio in at least one direction", "target_type": TargetType.LOWER_BOUND, "target_value":0}],
        "verb_descriptors":     [{"description": f"contracts in the {idx2component[2]['axis']} direction when the {idx2component[3]['axis']} direction is compressed", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[2]['axis']} direction when the {idx2component[3]['axis']} direction is compressed", "target_type": TargetType.LOWER_BOUND, "target_value":0},
                                 {"description": f"expands in the {idx2component[2]['axis']} direction when the {idx2component[3]['axis']} direction is stretched", "target_type": TargetType.UPPER_BOUND, "target_value":0},
                                 {"description": f"contracts in the {idx2component[2]['axis']} direction when the {idx2component[3]['axis']} direction is stretched", "target_type": TargetType.LOWER_BOUND, "target_value":0}]
    }, 

    # ----- Bulk Modulus Information -----
    'K': {
        "full_prop_name": "bulk modulus",
        "alternate_symbols": ["B"],
        "property_generality": PropertyGenerality.OVERALL,
        "property_type": PropertyType.BULK_MOD,
        "dataset_coverage": {
            "min": 0,
            "max": 3.5,
            "q1": 0.01,
            "q3": 0.09,
            "densely_populated_ranges": [[0, 0.2], [1.25, 1.75]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": "highly resistant to compression", "target_type": TargetType.LOWER_BOUND, "target_value":1.5},
                                 {"description": "resistant to compression", "target_type": TargetType.LOWER_BOUND, "target_value":1.0},
                                 {"description": "compressible", "target_type": TargetType.UPPER_BOUND, "target_value":0.5},
                                 {"description": "highly compressible", "target_type": TargetType.UPPER_BOUND, "target_value":0.2}],
        "property_descriptors": [{"description": "a very high bulk modulus", "target_type": TargetType.LOWER_BOUND, "target_value":1.5},
                                 {"description": "a high bulk modulus", "target_type": TargetType.LOWER_BOUND, "target_value":1.0},
                                 {"description": "a low bulk modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.5},
                                 {"description": "a very low bulk modulus", "target_type": TargetType.UPPER_BOUND, "target_value":0.2}]
    },

    # ----- Relative Density Information -----
    'V': {
        "full_prop_name": "volume fraction",
        "alternate_symbols": ["Relative density", "fill fraction"],
        "property_generality": PropertyGenerality.OVERALL,
        "property_type": PropertyType.VOLUME_FRACTION,
        "dataset_coverage": {
            "min": 0.002,
            "max": 1,
            "q1": 0.06,
            "q3": 0.2,
            "densely_populated_ranges": [[0, 0.4]]
        },
        "smallest_meaningful_quantization": 0.01,
        "adjective_descriptors":[{"description": "very dense", "target_type": TargetType.LOWER_BOUND, "target_value":0.7},
                                 {"description": "dense", "target_type": TargetType.LOWER_BOUND, "target_value":0.5},
                                 {"description": "sparse", "target_type": TargetType.UPPER_BOUND, "target_value":0.3},
                                 {"description": "very sparse", "target_type": TargetType.LOWER_BOUND, "target_value":0.2}],
        "property_descriptors": [{"description": "a very high volume fraction", "target_type": TargetType.LOWER_BOUND, "target_value":0.7},
                                 {"description": "a high volume fraction", "target_type": TargetType.LOWER_BOUND, "target_value":0.5},
                                 {"description": "a low volume fraction", "target_type": TargetType.UPPER_BOUND, "target_value":0.3},
                                 {"description": "a very low volume fraction", "target_type": TargetType.LOWER_BOUND, "target_value":0.2}]
    },
}

def score_properties_by_interest(propsIn:dict) -> dict:
    UNSCORED = 0
    HIGH_INTEREST = 1e3
    MEDIUM_INTEREST = 1e2
    SOME_INTEREST = 1e1
    DISCOURAGE = -3
    DO_NOT_USE = -1e5

    uai_threshold = 0.0025    # observed UAI value, below which there's no discernable difference in directional properties; use only global props

    scored_props = {}
    for p in propsIn.keys():
        scored_props[p] = {}
        scored_props[p]["usable"] = True
        scored_props[p]["score"] = UNSCORED

    # evaluate isotropy implications
    if propsIn['A'] < uai_threshold:              # if isotropic, mention it + prevent the use of directional properties
        scored_props['A']["score"] += HIGH_INTEREST
        for p in propsIn.keys():
            if property_references[p]["property_generality"] != PropertyGenerality.OVERALL:
                scored_props[p]["usable"] = False
                scored_props[p]["score"] = DO_NOT_USE
    else:                                         # check whether the directional properties have interesting differences
        # look at youngs mod 
        es = ['E_1', 'E_2', 'E_3']
        epairs = list(itertools.combinations(es, 2))
        for ep in epairs:
            # if one of the values is 0, use the other as the denominator
            denom = abs(propsIn[ep[0]])
            if denom == 0:
                denom == abs(propsIn[ep[1]])
            # if denom == 0 then ep[0] == ep[1] == 0, they are certainly not 20% different
            if denom > 0 and abs(propsIn[ep[0]] - propsIn[ep[1]]) / denom > 0.2: # values have a 20% difference or more
                scored_props[ep[0]]["score"] += MEDIUM_INTEREST
                scored_props[ep[1]]["score"] += MEDIUM_INTEREST

        # look at shear mod
        gs = ['G_23', 'G_31', 'G_12']
        gpairs = list(itertools.combinations(gs, 2))
        for gp in gpairs:
            denom = abs(propsIn[gp[0]])
            if denom == 0:
                denom == abs(propsIn[gp[1]])
            # if denom == 0 then gp[0] == gp[1] == 0, they are certainly not 20% different
            if denom > 0 and abs(propsIn[gp[0]] - propsIn[gp[1]]) / denom > 0.2: # values have a 20% difference or more
                scored_props[gp[0]]["score"] += MEDIUM_INTEREST
                scored_props[gp[1]]["score"] += MEDIUM_INTEREST

        # look at poisson ratio
        nus = [['nu_12', 'nu_21'],
               ['nu_13', 'nu_31'],
               ['nu_23', 'nu_32']]
        nupairs = list(itertools.combinations(nus, 2))
        for nup in nupairs:
            if abs(propsIn[nup[0][0]] - propsIn[nup[1][0]]) / propsIn[nup[0][0]] > 0.2 or \
               abs(propsIn[nup[0][1]] - propsIn[nup[1][1]]) / propsIn[nup[0][1]] > 0.2: # values have a 20% difference or more
                scored_props[nup[0][0]]["score"] += MEDIUM_INTEREST
                scored_props[nup[0][1]]["score"] += MEDIUM_INTEREST
                scored_props[nup[1][0]]["score"] += MEDIUM_INTEREST
                scored_props[nup[1][1]]["score"] += MEDIUM_INTEREST
        

    # if auxetic, mention it
    if propsIn['nu'] > 0:
        scored_props['nu']["score"] += DISCOURAGE
    elif propsIn['nu'] < 0:
        scored_props['nu']["score"] += HIGH_INTEREST

    # if low V and high E, mention it. Both in [0,1] so can directly look for something that preserves eg. c*E_{material} with 1/preservation_ratio*c*V
    preservation_ratio = 1.2
    if propsIn['E'] / propsIn['V'] >= preservation_ratio:
        scored_props['E']["score"] += HIGH_INTEREST
        scored_props['V']["score"] += HIGH_INTEREST

    # if out of distribution, increase score based on relative distance from q1/q3 and well-sampled ranges
    for p in propsIn.keys():
        val = propsIn[p]
        if not isinstance(val, float):
            continue
        dataset_info = property_references[p]["dataset_coverage"]
        dataset_span = abs(dataset_info["max"] - dataset_info["min"])
        # check whether it's outside the middle quartiles
        if val < dataset_info["q1"]: 
            relative_dist = abs(val - dataset_info["q1"]) / dataset_span  # further distance --> higher score
            scored_props[p]["score"] += MEDIUM_INTEREST * relative_dist
        if val > dataset_info["q3"]:
            relative_dist = abs(val - dataset_info["q3"]) / dataset_span  # further distance --> higher score
            scored_props[p]["score"] += MEDIUM_INTEREST * relative_dist\
        # check whether it's inside/outside a densely populated range    
        inside_dense_range = False
        closest_dist_to_range = 1e9
        for popd_range in dataset_info["densely_populated_ranges"]:
            if popd_range[0] < val and val < popd_range[1]:
                inside_dense_range = True
                break
            else:
                closest_dist_to_range = min(closest_dist_to_range, abs(val - popd_range[0]))
                closest_dist_to_range = min(closest_dist_to_range, abs(val - popd_range[1]))
        if not inside_dense_range:
            relative_dist = closest_dist_to_range / dataset_span  # further distance --> higher score
            scored_props[p]["score"] += MEDIUM_INTEREST * relative_dist
        # check whether it's near one of the extreme points
        if abs(val - dataset_info["min"]) < dataset_span*0.1:
            if val not in dataset_info["densely_populated_ranges"][0]: # reward this as long as it's near the extreme, and the extreme is not well represented
                relative_dist = abs(val - dataset_info["min"]) / dataset_span
                if relative_dist == 0:
                    scored_props[p]["score"] += HIGH_INTEREST * 10 # this is the extreme datapoint, we should definitely pick it
                else:
                    reward_factor = 1 / relative_dist # closer distance --> higher score
                    scored_props[p]["score"] += MEDIUM_INTEREST * reward_factor

    return scored_props

def get_random_valid_subset(propsIn:dict, min_num_active_props:int, max_num_active_props:int) -> list[str]:
    uai_threshold = 0.0025    # observed UAI value, below which there's no discernable difference in directional properties; use only global props
    probability_of_using_directional_props = 0.25 

    num_remaining_props = max_num_active_props

    # pick at most two of (G, K, nu, E) -- the rest are derivable (for isotropic materials, anyway; we're using this as a proxy)
    if random() < 0.9:
        num_moduli_to_include = randint(0, min(2, num_remaining_props))
    else:
        num_moduli_to_include = randint(0, num_remaining_props) # sometimes, you can pick more. just for the sake of profile diversity / might want this for more anisotropic materials

    related_moduli = [PropertyType.BULK_MOD, PropertyType.POISSON_RATIO, PropertyType.YOUNGS_MOD, PropertyType.SHEAR_MOD]
    prop_types_to_use = sample(related_moduli, num_moduli_to_include) if num_moduli_to_include > 0 else []
    num_remaining_props -= num_moduli_to_include

    # add in the other props
    other_props = [p for p in PropertyType if p not in related_moduli]
    min_to_add = max(len(prop_types_to_use) - min_num_active_props, (0 if num_moduli_to_include > 1 else 1))
    num_other_props_to_include = randint(min_to_add, max(num_remaining_props, len(other_props)))
    if num_other_props_to_include > 0:
        prop_types_to_use.extend(sample(other_props, num_other_props_to_include))

    num_remaining_props -= len(prop_types_to_use)

    # figure out whether to use directional or overall measures for each active property
    allow_directional_props = propsIn['A'] > uai_threshold 
    
    propIn_keys = [k for k in propsIn.keys()]
    prop_set_to_use = []
    for prop_type in prop_types_to_use:
        # find all the props of this type
        prop_cand_keys = [prop_key for prop_key in propIn_keys if property_references[prop_key]["property_type"] == prop_type]
        overall_cand_keys = [prop_key for prop_key in prop_cand_keys if property_references[prop_key]["property_generality"] == PropertyGenerality.OVERALL]
        directional_cand_keys = [prop_key for prop_key in prop_cand_keys if property_references[prop_key]["property_generality"] != PropertyGenerality.OVERALL]

        # figure out whether to use overall or directional properties
        use_directional_keys = allow_directional_props and len(directional_cand_keys) > 0 and random() < probability_of_using_directional_props

        if not use_directional_keys: # get an overall measure for this property (no directional)
            idx = randint(0, len(overall_cand_keys)-1) if len(overall_cand_keys) > 1 else 0
            key_to_add = overall_cand_keys[idx]
            prop_set_to_use.append(key_to_add)
        else:
            # if nu, make sure we only use non-equivalent directional specs
            if prop_type == PropertyType.POISSON_RATIO:
                equiv_directional_values = [[PropertyGenerality.DIRECTIONAL_12, PropertyGenerality.DIRECTIONAL_21],
                                            [PropertyGenerality.DIRECTIONAL_13, PropertyGenerality.DIRECTIONAL_31],
                                            [PropertyGenerality.DIRECTIONAL_23, PropertyGenerality.DIRECTIONAL_32]]
                directional_value_cands = []
                for pair in equiv_directional_values:
                    idx = 0 if random() < 0.5 else 1
                    directional_value_cands.append(pair[idx])
                directional_cand_subset = [prop_key for prop_key in directional_cand_keys if property_references[prop_key]["property_generality"] in directional_value_cands]
                directional_cand_keys = directional_cand_subset

            # pick a subset of the directional keys to keep 
            num_to_keep = randint(1,len(directional_cand_keys)) if len(directional_cand_keys) > 1 else 1
            keys_to_add = [directional_cand_keys[i] for i in sample(range(len(directional_cand_keys)), num_to_keep)]
            prop_set_to_use.extend(keys_to_add)
    return prop_set_to_use

def get_valid_subset_using_scores(scoredProps:dict, min_num_active_props:int, max_num_active_props:int) -> list[str]:
    active_props = []

    related_moduli = [PropertyType.BULK_MOD, PropertyType.POISSON_RATIO, PropertyType.YOUNGS_MOD, PropertyType.SHEAR_MOD] # generally want at most 2 of these in a given profile, at least for isotropic the rest can be derived
    equiv_directional_values = [[PropertyGenerality.DIRECTIONAL_12, PropertyGenerality.DIRECTIONAL_21], # only want at most one of each inner list, since they're closely related
                                [PropertyGenerality.DIRECTIONAL_13, PropertyGenerality.DIRECTIONAL_31],
                                [PropertyGenerality.DIRECTIONAL_23, PropertyGenerality.DIRECTIONAL_32]]
    num_active_props_from_related_moduli = 0

    ordered_prop_keys = sorted(scoredProps, key=lambda k: scoredProps[k]["score"], reverse=True) # gives highest-ranked elements first
    for cand_prop_key in ordered_prop_keys:
        if scoredProps[cand_prop_key]["score"] <= 0: # unscored or negatively scored aren't worth sorting
            break

        if not scoredProps[cand_prop_key]["usable"]: # skip this property if it was marked DO_NOT_USE at some point
            continue

        cand_ok_to_add = True
        # check if something else of this property_type is already in our profile
        curr_prop_key_type = property_references[cand_prop_key]["property_type"]
        curr_prop_key_generality = property_references[cand_prop_key]["property_generality"]
        active_props_of_same_type = [pkey for pkey in active_props if property_references[pkey]["property_type"] == curr_prop_key_type]
        for ap_key in active_props_of_same_type:
            # if either is overall, skip.
            if property_references[ap_key]["property_generality"] == PropertyGenerality.OVERALL or curr_prop_key_generality == PropertyGenerality.OVERALL:
                cand_ok_to_add = False
                break
            # check for duplication in the nu case
            if curr_prop_key_type == PropertyType.POISSON_RATIO:
                for equiv_pair in equiv_directional_values:
                    if curr_prop_key_generality in equiv_pair and property_references[ap_key]["property_generality"] in equiv_pair:
                        cand_ok_to_add = False
                        break
        if not cand_ok_to_add:
            continue # skip to new cand_prop_key

        # check if we're adding too many related_moduli
        if num_active_props_from_related_moduli >= 2 and curr_prop_key_type in related_moduli:
            if random() < 0.6: # throw out most of the time
                cand_ok_to_add = False
                continue   # skip to new cand_prop_key

        # passed all the checks, add the property
        if cand_ok_to_add:
            active_props.append(cand_prop_key)
        
        # if we've hit our max num properties, break
        if len(active_props) >= max_num_active_props:
            break
        elif len(active_props) > min_num_active_props:
            if random() < 0.1: # terminate early
                break

    # if we broke out early, complete the profile with some random elements if desired
    if len(active_props) < max_num_active_props:
        num_open_spots = max_num_active_props - len(active_props)
        num_to_add = randint(0, num_open_spots)
        if len(active_props) < min_num_active_props:
            num_to_add = max(num_to_add, min_num_active_props - len(active_props))
        if num_to_add > 0:
            remaining_props = [pkey for pkey in scoredProps.keys() if (pkey not in active_props and scoredProps[pkey]["usable"])]
            props_to_add = sample(remaining_props, num_to_add)
            active_props.extend(props_to_add)

    return active_props

def get_property_subset(propsIn:dict, min_num_active_props:int, max_num_active_props:int) -> list:
    use_ranked_properties = True

    # pick out the types of properties to include 
    if use_ranked_properties:
        scored_props = score_properties_by_interest(propsIn)
        prop_set_to_use = get_valid_subset_using_scores(scored_props, min_num_active_props, max_num_active_props)
    else:
        prop_set_to_use = get_random_valid_subset(propsIn, min_num_active_props, max_num_active_props)

    return prop_set_to_use

def input_satisfies_description__float(prop_value:float, target_type:TargetType, target_value:float) -> bool:
    match target_type:
        case TargetType.VALUE:
            approx_prop_value = round_to_1_sig_fig(prop_value)
            return abs(target_value - approx_prop_value) <= abs(0.1*target_value)
        case TargetType.LOWER_BOUND:
            return prop_value >= target_value
        case TargetType.UPPER_BOUND:
            return prop_value <= target_value
        case TargetType.RANGE:
            raise NotImplementedError()
        case TargetType.CATEGORICAL:
            raise Exception("Not a float type")
        case _:
            raise Exception("Invalid target type")
        
def format_numerical_target_descriptor(prop_name:str, target_type:TargetType, target_value:float) -> str:
    match target_type:
        case TargetType.VALUE:
            return f"{prop_name} = {target_value}"
        case TargetType.LOWER_BOUND:
            return f"{prop_name} >= {target_value}"
        case TargetType.UPPER_BOUND:
            return f"{prop_name} <= {target_value}"
        case TargetType.RANGE:
            raise f"{target_value[0]} <= {prop_name} <= {target_value[1]}"
        case TargetType.CATEGORICAL:
            raise Exception("Not a float type")
        case _:
            raise Exception("Invalid target type")
        
class PropertyTargetSelector(Enum):
    RANDOM = 0
    TIGHTEST = 1
    ALL = 2

def construct_all_suitable_profiles_for_prop(propIn_key:str, propIn_value:float | bool, num_targets_to_keep:int, target_types_to_keep:PropertyTargetSelector) -> list[dict]:
    # decide which of the descriptions we satisfy, and organize based on the target value and type
    suitable_target_values = {}

    def update_suitable_target_dict(cand_targ_val, cand_targ_type, info):
        # add to the dictionary
        if cand_targ_val in suitable_target_values:
            if cand_targ_type in suitable_target_values[cand_targ_val]:
                suitable_target_values[cand_targ_val][cand_targ_type].append(info)
            else:
                suitable_target_values[cand_targ_val][cand_targ_type] = [info]
        else:
            suitable_target_values[cand_targ_val] = {}
            suitable_target_values[cand_targ_val][cand_targ_type] = [info]

    descriptor_collections = [{"collection_name": "adjective_descriptors", "collection_keyword": "adjective"},
                              {"collection_name": "property_descriptors", "collection_keyword": "property"},
                              {"collection_name": "verb_descriptors", "collection_keyword": "verb"}]

    for collection in descriptor_collections:
        coll_name = collection["collection_name"]
        coll_kw = collection["collection_keyword"]

        if coll_name not in property_references[propIn_key]: # current property doesn't have any descriptions from this part of speech
            continue

        for description in property_references[propIn_key][coll_name]:
            cand_targ_val = description["target_value"]
            cand_targ_type = description["target_type"]
            if input_satisfies_description__float(propIn_value, cand_targ_type, cand_targ_val):
                cand_info = {"description": f"{description["description"]} ({format_numerical_target_descriptor(propIn_key, cand_targ_type, cand_targ_val)})",
                            "description_type": coll_kw}
                update_suitable_target_dict(cand_targ_val, cand_targ_type, cand_info)

    # add the "rougly equal to" case (suitable for any input)
    prop_name = property_references[propIn_key]["full_prop_name"]
    approx_prop_value = round_to_1_sig_fig(propIn_value)
    equivalence_description = {"description": f"a {prop_name} ({propIn_key}) of roughly {approx_prop_value}",
                               "description_type": "property"}
    update_suitable_target_dict(approx_prop_value, TargetType.VALUE, equivalence_description)

    # figure out which profiles to keep, if trimming is desired
    if target_types_to_keep == PropertyTargetSelector.RANDOM:
        cases_to_keep = sample([suitable_target_values.keys()], num_targets_to_keep)
    elif target_types_to_keep == PropertyTargetSelector.TIGHTEST:
        sorted_cases = sorted(suitable_target_values, key=lambda k: abs(k - propIn_value), reverse=True) # gives highest-ranked elements first
        cases_to_keep = sorted_cases[0:num_targets_to_keep]
    elif target_types_to_keep == PropertyTargetSelector.ALL:
        cases_to_keep = suitable_target_values.keys()
    else:
        raise Exception("Unsupported target selector")

    # create the final profiles corresponding to each target value & type
    suitable_target_profiles = []
    for targ_val in suitable_target_values.keys():
        if targ_val not in cases_to_keep:
            continue
        for targ_type in suitable_target_values[targ_val].keys():
            details = {}
            details["property"] = propIn_key
            details["target_value"] = targ_val
            details["target_type"] = targ_type.name.lower()
            details["target_descriptions"] = suitable_target_values[targ_val][targ_type] # all descriptions applicable to this value/type pair
            suitable_target_profiles.append(details)

    return suitable_target_profiles


def get_target_property_profile_from_properties(propertiesIn:dict, num_property_subsets:int, min_num_active_props:int, max_num_active_props:int, target_selector:PropertyTargetSelector=PropertyTargetSelector.TIGHTEST, num_targets_to_keep:int=1) -> list[list[dict]]:
    profiles = []
    for prof_num in range(num_property_subsets):
        prop_keys_to_use = get_property_subset(propertiesIn, min_num_active_props, max_num_active_props)

        possible_profile_elements = []
        for prop_key in prop_keys_to_use:
            possible_property_profiles = construct_all_suitable_profiles_for_prop(prop_key, propertiesIn[prop_key], num_targets_to_keep, target_selector)
            possible_profile_elements.append(possible_property_profiles)

        # create all the valid profile combinations
        profile_opts = itertools.product(*possible_profile_elements)
        profiles.extend([list(prof) for prof in profile_opts])

    return profiles
