"""
Utilities to handle our dataset working directory structure.
Works with both local paths and S3 paths.
"""

import os
import shutil
import logging
import tempfile
import boto3

class Workdir:
    root = '/metagen-data'
    s3 = False
    
    @classmethod
    def set_root(cls, root):
        cls.root = root
        cls.s3 = root.startswith('s3://')
    
    def abs_path(cls, path):
        return cls.root.rstrip('/') + '/' + path.lstrip('/')

    @classmethod
    def exists(cls, path):
        if cls.s3:
            return cls._s3_exists(path)
        else:
            return os.path.exists(path)
    
    @classmethod
    def touch(cls, path):
        if path.startswith('s3://'):
            cls._s3_touch(path)
        else:
            cls._local_touch(path)
    
    @classmethod
    def _local_touch(cls, path):
        with open(path, 'a'):
            os.utime(path, None)
    
    @classmethod
    def _s3_touch(cls, path):
        s3 = boto3.client('s3')
        s3.put_object(Bucket=path.split('/')[2], Key='/'.join(path.split('/')[3:]))
    
    @classmethod
    def _s3_exists(cls, path):
        s3 = boto3.client('s3')
        try:
            s3.head_object(Bucket=path.split('/')[2], Key='/'.join(path.split('/')[3:]))
            return True
        except:
            return False
    
