#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
<scene>/interpolate_novel_view_*/ours_30000/dvs_input/info.txt
→ events written / normalised to:
<scene>/interpolate_novel_view_*/ours_30000/dvs_output/events_2400fps_416us.txt
"""

import argparse
from pathlib import Path

import cv2
import numpy as np
import tqdm

from simulator.config import cfg
from simulator.simulator import EventSim


# ----------------------------- CLI ---------------------------------- #
def get_args():
    p = argparse.ArgumentParser()
    p.add_argument("--camera_type", default="DVS346")
    p.add_argument("--model_para", type=float, nargs="+")
    p.add_argument(
        "--root_dir",
        default="/share/magic_group/gs2e_data/output_more",
        help="Root that contains all scenes",
    )
    p.add_argument(
        "--scene_dir",
        default=None,
        help="Process a single scene (optional; will search under this dir)",
    )
    return p.parse_args()


def integrate_cfg(cfg, args):
    cfg.SENSOR.CAMERA_TYPE = args.camera_type
    cfg.SENSOR.K = args.model_para if args.model_para else cfg.SENSOR.K
    if cfg.SENSOR.K is None or len(cfg.SENSOR.K) != 6:
        raise ValueError("Camera intrinsic K must be 6 numbers")


# ------------------------- scene discovery -------------------------- #
def discover_scenes(root_dir):
    """Yield absolute paths to every info.txt under any novel‐view directory."""
    pattern = "edited_speed_adaptive/ours_30000/dvs_input/info.txt"
    for info_path in Path(root_dir).rglob(pattern):
        yield info_path.resolve()


# ------------------------- core processing -------------------------- #
def process_info_file(info_path: Path):
    info_path = info_path.resolve()
    scene_name = info_path.parents[3].name  # e.g. 0000a44a
    input_dir = info_path.parent
    output_dir = input_dir.parent / "dvs_output"
    output_dir.mkdir(exist_ok=True)

    # unified filenames
    plural_file  = output_dir / "events_2400fps_416us.txt"
    singular_new = output_dir / "event_2400fps_416us.txt"
    raw_file     = output_dir / "events.txt"

    # quick-exit chain
    if plural_file.exists():
        print(f"[{scene_name}] Found {plural_file.name} — skipping simulation.")
        return

    if singular_new.exists():
        singular_new.rename(plural_file)
        print(f"[{scene_name}] Renamed {singular_new.name} → {plural_file.name}")
        return

    if raw_file.exists():
        raw_file.rename(plural_file)
        print(f"[{scene_name}] Renamed events.txt → {plural_file.name}")
        return

    # normal simulation path
    print(f"[{scene_name}] Generating events → {plural_file.name}")

    # read frame list & timestamps
    with open(info_path, "r") as f:
        lines = [ln.strip() for ln in f if ln.strip()]
    timestamps  = [int(ln.split()[1]) for ln in lines]
    frame_paths = [ln.split()[0] for ln in lines]

    # set up simulator
    sim = EventSim(cfg=cfg, output_folder=str(output_dir), video_name=scene_name)

    events_accum = []
    pbar = tqdm.tqdm(total=len(frame_paths), desc=scene_name)
    for img_path, ts in zip(frame_paths, timestamps):
        img = cv2.imread(img_path, cv2.IMREAD_GRAYSCALE)
        ev  = sim.generate_events(img, ts)
        if ev is not None:
            events_accum.append(ev)
        total_events = sum(e.shape[0] for e in events_accum)
        pbar.set_description(f"{scene_name} | events: {total_events:,}")
        pbar.update(1)
    pbar.close()

    if events_accum:
        events = np.concatenate(events_accum, axis=0)
        np.savetxt(plural_file, events, fmt="%d")
        print(f"[{scene_name}] Event file saved: {plural_file}")
    else:
        print(f"[{scene_name}] Warning: no events generated")

    sim.reset()


# ----------------------------- main --------------------------------- #
if __name__ == "__main__":
    args = get_args()
    integrate_cfg(cfg, args)

    # decide search root: single‐scene or batch
    search_root = args.scene_dir if args.scene_dir is not None else args.root_dir
    all_infos = list(discover_scenes(search_root))
    if not all_infos:
        raise FileNotFoundError(f"No info.txt found under {search_root!r}")

    # process each discovered info.txt with progress bar
    with tqdm.tqdm(total=len(all_infos), desc="Scenes") as scene_bar:
        for info_txt in all_infos:
            process_info_file(info_txt)
            scene_bar.update(1)
