#
# Copyright (C) 2023, Inria
# GRAPHDECO research group, https://team.inria.fr/graphdeco
# All rights reserved.
#
# This software is free for non-commercial, research and evaluation use 
# under the terms of the LICENSE.md file.
#
# For inquiries contact  george.drettakis@inria.fr
#

import torch
from torch import nn
import numpy as np
from gaussian_splatting.utils.graphics_utils import getWorld2View2, getProjectionMatrix
from gaussian_splatting.utils.general_utils import PILtoTorch
import cv2

class Camera(nn.Module):
    def __init__(self, resolution, colmap_id, R, T, FoVx, FoVy, depth_params, image, invdepthmap,
                 image_name, uid,
                 trans=np.array([0.0, 0.0, 0.0]), scale=1.0, data_device = "cuda",
                 train_test_exp = False, is_test_dataset = False, is_test_view = False
                 ):
        super(Camera, self).__init__()

        self.uid = uid
        self.colmap_id = colmap_id
        self.R = R 
        self.T = T
        self.FoVx = FoVx
        self.FoVy = FoVy
        self.image_name = image_name
        self.image_pil = image
        self.is_interp = False

        try:
            self.data_device = torch.device(data_device)
        except Exception as e:
            print(e)
            print(f"[Warning] Custom device {data_device} failed, fallback to default cuda device" )
            self.data_device = torch.device("cuda")

        resized_image_rgb = PILtoTorch(image, resolution)
        gt_image = resized_image_rgb[:3, ...]
        self.alpha_mask = None
        if resized_image_rgb.shape[0] == 4:
            self.alpha_mask = resized_image_rgb[3:4, ...].to(self.data_device)
        else: 
            self.alpha_mask = torch.ones_like(resized_image_rgb[0:1, ...].to(self.data_device))

        if train_test_exp and is_test_view:
            if is_test_dataset:
                self.alpha_mask[..., :self.alpha_mask.shape[-1] // 2] = 0
            else:
                self.alpha_mask[..., self.alpha_mask.shape[-1] // 2:] = 0

        self.original_image = gt_image.clamp(0.0, 1.0).to(self.data_device)
        self.image_width = self.original_image.shape[2]
        self.image_height = self.original_image.shape[1]

        self.invdepthmap = None
        self.depth_reliable = False
        if invdepthmap is not None:
            self.depth_mask = torch.ones_like(self.alpha_mask)
            self.invdepthmap = cv2.resize(invdepthmap, resolution)
            self.invdepthmap[self.invdepthmap < 0] = 0
            self.depth_reliable = True

            if depth_params is not None:
                if depth_params["scale"] < 0.2 * depth_params["med_scale"] or depth_params["scale"] > 5 * depth_params["med_scale"]:
                    self.depth_reliable = False
                    self.depth_mask *= 0
                
                if depth_params["scale"] > 0:
                    self.invdepthmap = self.invdepthmap * depth_params["scale"] + depth_params["offset"]

            if self.invdepthmap.ndim != 2:
                self.invdepthmap = self.invdepthmap[..., 0]
            self.invdepthmap = torch.from_numpy(self.invdepthmap[None]).to(self.data_device)

        self.zfar = 100.0
        self.znear = 0.01

        self.trans = trans
        self.scale = scale

        # 视图矩阵，世界坐标系变换到相机坐标系
        # self.world_view_transform = torch.tensor(getWorld2View2(R, T, trans, scale)).transpose(0, 1).cuda()
        self.world_view_transform = torch.tensor(getWorld2View2(R, T, trans, scale)).transpose(0, 1).cuda()
        # 投影矩阵
        self.projection_matrix = getProjectionMatrix(znear=self.znear, zfar=self.zfar, fovX=self.FoVx, fovY=self.FoVy).transpose(0,1).cuda()
        # 合并的变换
        self.full_proj_transform = (self.world_view_transform.unsqueeze(0).bmm(self.projection_matrix.unsqueeze(0))).squeeze(0)
        # 相机中心
        self.camera_center = self.world_view_transform.inverse()[3, :3]
        
class MiniCam:
    def __init__(self, width, height, fovy, fovx, znear, zfar, world_view_transform, full_proj_transform):
        self.image_width = width
        self.image_height = height    
        self.FoVy = fovy
        self.FoVx = fovx
        self.znear = znear
        self.zfar = zfar
        self.world_view_transform = world_view_transform
        self.full_proj_transform = full_proj_transform
        view_inv = torch.inverse(self.world_view_transform)
        self.camera_center = view_inv[3][:3]

class LightCam:
    """
    轻量级相机类：可从训练相机中提取必要信息，用于插值与渲染。
    保留内参 + 位姿（旋转 + 中心）+ 场景变换参数，支持延迟计算矩阵。
    """
    def __init__(
        self,
        image_width: int,
        image_height: int,
        FoVx: float,
        FoVy: float,
        znear: float = 0.01,
        zfar: float = 100.0,
        R: torch.Tensor = None,         # (3, 3) 旋转矩阵
        T: torch.Tensor = None,         # (3,) colmap 平移向量
        trans: torch.Tensor = None,     # scene 归一化平移
        scale: float = 1.0,             # scene 归一化尺度
        device: str = "cuda",
        uid: int = -1,
        image_name: str = "",
    ):
        self.image_width = image_width
        self.image_height = image_height
        self.FoVx = FoVx
        self.FoVy = FoVy
        self.znear = znear
        self.zfar = zfar

        self.uid = uid
        self.image_name = image_name
        self.device = device

        self.R = R.to(device) if R is not None else torch.eye(3, device=device)
        self.T = T.to(device) if T is not None else torch.zeros(3, device=device)
        self.trans = trans.to(device) if trans is not None else torch.zeros(3, device=device)
        self.scale = scale

        # 延迟计算变量
        self._C = None
        self._world_view_transform = None
        self._full_proj_transform = None

    @property
    def camera_center(self):
        if self._C is None:
            self._C = -torch.inverse(self.R) @ self.T  # C = -R^{-1} t
        return self._C

    @property
    def world_view_transform(self):
        if self._world_view_transform is None:
            from gaussian_splatting.utils.graphics_utils import getWorld2View2
            self._world_view_transform = torch.tensor(
                getWorld2View2(self.R.cpu().numpy(), self.T.cpu().numpy(), self.trans.cpu().numpy(), self.scale),
                dtype=torch.float32, device=self.device
            ).T
        return self._world_view_transform

    @property
    def full_proj_transform(self):
        if self._full_proj_transform is None:
            from gaussian_splatting.utils.graphics_utils import getProjectionMatrix
            P = torch.tensor(
                getProjectionMatrix(self.znear, self.zfar, self.FoVx, self.FoVy),
                dtype=torch.float32, device=self.device
            ).T
            self._full_proj_transform = self.world_view_transform.unsqueeze(0).bmm(P.unsqueeze(0)).squeeze(0)
        return self._full_proj_transform

    def to(self, device: str):
        self.device = device
        self.R = self.R.to(device)
        self.T = self.T.to(device)
        self.trans = self.trans.to(device)
        if self._C is not None:
            self._C = self._C.to(device)
        if self._world_view_transform is not None:
            self._world_view_transform = self._world_view_transform.to(device)
        if self._full_proj_transform is not None:
            self._full_proj_transform = self._full_proj_transform.to(device)
        return self
