import os
import subprocess
import argparse
from pathlib import Path

# === 配置路径 ===
MODEL_ROOT = Path("/share/magic_group/gs2e_data/output_more")
SOURCE_ROOT = Path("/share/magic_group/gs2e_data/source_more")
SCRIPT_PATH = Path("/share/magic_group/GS2Event-Simulator/_1_render_interpolate.py")
LOCK_FILENAME = ".rendering.lock"


def is_rendered(scene_name: str) -> bool:
    """判断渲染是否已完成：检查 pose_esim.txt 是否存在"""
    pose_file = MODEL_ROOT / scene_name / "edited_speed_adaptive" / "ours_30000" / "esim" / "pose_esim.txt"
    return pose_file.is_file()


def acquire_lock(scene_name: str) -> bool:
    """
    尝试为场景创建锁文件。
    成功返回 True，已存在锁文件返回 False。
    """
    lock_path = MODEL_ROOT / scene_name / LOCK_FILENAME
    try:
        # 原子创建
        fd = os.open(lock_path, os.O_CREAT | os.O_EXCL | os.O_WRONLY)
        os.close(fd)
        return True
    except FileExistsError:
        return False


def release_lock(scene_name: str):
    """删除场景的锁文件"""
    lock_path = MODEL_ROOT / scene_name / LOCK_FILENAME
    try:
        lock_path.unlink()
    except FileNotFoundError:
        pass


def make_command(scene_name: str, iteration: int) -> list:
    model_path = MODEL_ROOT / scene_name
    source_path = SOURCE_ROOT / scene_name
    return [
        "python", str(SCRIPT_PATH),
        "--model_path", str(model_path),
        "--source_path", str(source_path),
        "--iteration", str(iteration),
        "--skip_test",
        "--interpolate",
        "--interp_mode", "ada_speed",
        # "--novel_view",
        "--nums_inserted", "5"
    ]


def main(iteration: int):
    scenes = sorted([d.name for d in MODEL_ROOT.iterdir() if d.is_dir()])
    for scene in scenes:
        # 如果已经渲染完成，跳过
        if is_rendered(scene):
            continue
        # 尝试获取锁，防止多个脚本实例同时处理同一个场景
        if not acquire_lock(scene):
            continue
        try:
            cmd = make_command(scene, iteration)
            print(f"Rendering scene {scene}...")
            subprocess.run(cmd, check=True)
        except subprocess.CalledProcessError as e:
            print(f"Error rendering {scene}: {e}")
        finally:
            # 无论成功失败都释放锁
            release_lock(scene)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Serial Render Launcher")
    parser.add_argument("--iteration", type=int, default=30000,
                        help="Model iteration to load (e.g., 30000)")
    args = parser.parse_args()
    main(args.iteration)
