#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
gen_info.py — 生成 info.txt（图片绝对路径  +  微秒时间戳）

用法
----
python gen_info.py /path/to/images --fps 30
"""
import argparse
import sys
import re
from pathlib import Path

# --------- “自然数”排序工具（让 2 排在 10 前） ------------------
_num_pat = re.compile(r'(\d+)')
def _natural_key(s: str):
    return [int(t) if t.isdigit() else t.lower() for t in _num_pat.split(s)]

# -----------------------------------------------------------------
def main(root: Path, fps: float):
    if not root.is_dir():
        sys.exit(f"[错误] {root} 不是合法目录。")

    exts = {'.jpg', '.jpeg', '.png', '.bmp', '.tif', '.tiff'}
    imgs = [p for p in root.iterdir() if p.suffix.lower() in exts]
    if not imgs:
        sys.exit("[错误] 该目录下未找到图片文件。")

    imgs.sort(key=lambda p: _natural_key(p.name))

    delta_us = int(1_000_000 / fps)
    info_path = root / "info.txt"
    with open(info_path, "w") as f:
        for idx, img in enumerate(imgs):
            ts = idx * delta_us
            # ★ 关键改动：使用绝对路径
            f.write(f"{img.resolve()} {ts}\n")

    print(f"[完成] 已向 {info_path} 写入 {len(imgs)} 条记录。")

# -----------------------------------------------------------------
if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="按顺序读取目录内图片，生成 info.txt（绝对路径 + 微秒时间戳）。")
    parser.add_argument("root_dir", type=Path, help="包含图片的目录")
    parser.add_argument("--fps", type=float, default=2400,
                        help="帧率 (frames per second)，默认 30")
    args = parser.parse_args()
    main(args.root_dir, args.fps)
