#! python3
# -*- encoding: utf-8 -*-
'''
@File    :   main.py
@Time    :   2022/7/12 17:30
@Author  :   Songnan Lin, Ye Ma
@Contact :   songnan.lin@ntu.edu.sg, my17@tsinghua.org.cn
@Note    :   
@inproceedings{lin2022dvsvoltmeter,
  title={DVS-Voltmeter: Stochastic Process-based Event Simulator for Dynamic Vision Sensors},
  author={Lin, Songnan and Ma, Ye and Guo, Zhenhua and Wen, Bihan},
  booktitle={ECCV},
  year={2022}
}
'''

import argparse
import os
import numpy as np
import cv2
import tqdm
from simulator.config import cfg
from simulator.simulator import EventSim
from simulator.visualize import events_to_voxel_grid, visual_voxel_grid

def get_args_from_command_line():
    parser = argparse.ArgumentParser(description='Parser of Runner of Network')
    parser.add_argument('--camera_type', type=str, help='Camera type, such as DVS346', default='DVS346')
    parser.add_argument('--model_para', type=float, nargs='+', help='Set parameters for a specific camera type', default=None)
    parser.add_argument('--input_dir', type=str, help='Set dataset root_path', default=None)
    parser.add_argument('--output_dir', type=str, help='Set output path', default=None)
    args = parser.parse_args()
    return args

def integrate_cfg(cfg, command_line_args):
    args = command_line_args
    cfg.SENSOR.CAMERA_TYPE = args.camera_type if args.camera_type is not None else cfg.SENSOR.CAMERA_TYPE
    cfg.SENSOR.K = args.model_para if args.model_para is not None else cfg.SENSOR.K
    cfg.DIR.IN_PATH = args.input_dir if args.input_dir is not None else cfg.DIR.IN_PATH
    cfg.DIR.OUT_PATH = args.output_dir if args.output_dir is not None else cfg.DIR.OUT_PATH
    if cfg.SENSOR.K is None or len(cfg.SENSOR.K) != 6:
        raise Exception('No model parameters given for sensor type %s' % cfg.SENSOR.CAMERA_TYPE)
    print(cfg)
    return cfg

def is_valid_dir(dirs):
    return os.path.exists(os.path.join(dirs, 'info.txt'))

def process_dir(cfg, file_info, video_name):
    indir = os.path.join(cfg.DIR.IN_PATH, video_name)
    outdir = os.path.join(cfg.DIR.OUT_PATH, video_name)
    print(f"Processing folder {indir}... Generating events in file {outdir}")

    # file info
    file_timestamps_us = [int(info_i.split()[1]) for info_i in file_info]
    file_paths = [info_i.split()[0] for info_i in file_info]

    # set simulator
    sim = EventSim(cfg=cfg, output_folder=cfg.DIR.OUT_PATH, video_name=video_name)

    # process
    pbar = tqdm.tqdm(total=len(file_paths))
    num_events, num_on_events, num_off_events = 0, 0, 0
    events = []
    for i in range(0, len(file_paths)):
        timestamp_us = file_timestamps_us[i]
        image = cv2.imread(file_paths[i], cv2.IMREAD_GRAYSCALE)

        # event generation!!!
        event = sim.generate_events(image, timestamp_us)

        if event is not None:
            events.append(event)
            num_events += event.shape[0]
            num_on_events += np.sum(event[:, -1] == 1)
            num_off_events += np.sum(event[:, -1] == 0)

        pbar.set_description(f"Events generated: {num_events}")
        pbar.update(1)

    events = np.concatenate(events, axis=0)
    np.savetxt(os.path.join(cfg.DIR.OUT_PATH, video_name + '.txt'), events, fmt='%1.0f')
    sim.reset()

def simulate_events_from_info_txt(
    info_txt_path: str,
    output_txt_path: str,
    camera_type="DVS346",
    model_params=[0.00018 * 29250, 20, 0.0001, 1e-7, 5e-9, 0.00001]
):
    """
    Runs DVS-Voltmeter simulation from a standalone info.txt file.
    Each line in info.txt: <absolute_path_to_image> <timestamp_us>
    """
    from types import SimpleNamespace
    from simulator.config import cfg as dvs_cfg
    from simulator.api import integrate_cfg
    from simulator.simulator import EventSim
    import tqdm
    import cv2
    import numpy as np
    import os

    video_name = os.path.splitext(os.path.basename(info_txt_path))[0]

    args = SimpleNamespace(
        camera_type=camera_type,
        model_para=model_params,
        input_dir="unused",   # no longer used
        output_dir=os.path.dirname(output_txt_path)
    )

    cfg = integrate_cfg(dvs_cfg, args)
    sim = EventSim(cfg=cfg, output_folder=args.output_dir, video_name=video_name)

    with open(info_txt_path, 'r') as f:
        lines = f.readlines()

    file_paths = [line.split()[0] for line in lines]
    file_timestamps_us = [int(line.split()[1]) for line in lines]

    events = []
    num_events = 0
    num_on_events = 0
    num_off_events = 0

    pbar = tqdm.tqdm(total=len(file_paths), desc=f"Events: 0")
    for i in range(len(file_paths)):
        image = cv2.imread(file_paths[i], cv2.IMREAD_GRAYSCALE)
        timestamp_us = file_timestamps_us[i]
        e = sim.generate_events(image, timestamp_us)

        if e is not None:
            events.append(e)
            num_events += e.shape[0]
            num_on_events += np.sum(e[:, -1] == 1)
            num_off_events += np.sum(e[:, -1] == 0)

            # === 保存每帧事件为 .npy 文件 ===
            npy_path = os.path.join(os.path.dirname(output_txt_path), f"{i:05d}.npy")
            np.save(npy_path, e.astype(np.int32))  # 存储为整数格式，节省空间

        # 设置动态进度条说明
        pbar.set_description(f"Events: {num_events:,}")
        pbar.update(1)


    sim.reset()
    pbar.close()

    os.makedirs(os.path.dirname(output_txt_path), exist_ok=True)
    if events:
        events = np.concatenate(events, axis=0)
        np.savetxt(output_txt_path, events, fmt='%1.0f')
        print(f"[✔] Saved events to: {output_txt_path}")
    else:
        print("[⚠️] No events generated!")



if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Single-scene DVS Simulation with configurable paths')
    parser.add_argument('--camera_type', type=str, default='DVS346', help='Camera type, e.g. DVS346')
    parser.add_argument('--model_para', type=float, nargs=6, help='6 model parameters to override config, optional')
    parser.add_argument('--info_txt', type=str, required=True, help='Path to info.txt file')
    parser.add_argument('--out_txt', type=str, required=True, help='Path to save generated events')

    args = parser.parse_args()

    # === Step 1: 保留原始 config 加载流程 ===
    from simulator.config import cfg
    cfg.SENSOR.CAMERA_TYPE = args.camera_type
    if args.model_para:
        cfg.SENSOR.K = args.model_para

    print(cfg)  # 可选：用于调试

    # === Step 2: 加载 info.txt ===
    video_name = os.path.splitext(os.path.basename(args.info_txt))[0]
    with open(args.info_txt, 'r') as f:
        file_info = f.readlines()

    # === Step 3: 调用原始 process_dir（保持不变） ===
    # 注意：cfg.DIR.OUT_PATH 只用于构造 simulator 的输出目录，不影响最终事件存储
    cfg.DIR.OUT_PATH = os.path.dirname(args.out_txt)
    process_dir(cfg=cfg, file_info=file_info, video_name=video_name)

    # === Step 4: 把事件另存为你指定的路径 ===
    src_events_path = os.path.join(cfg.DIR.OUT_PATH, video_name + ".txt")
    if src_events_path != args.out_txt:
        import shutil
        os.makedirs(os.path.dirname(args.out_txt), exist_ok=True)
        shutil.move(src_events_path, args.out_txt)
        print(f"[✔] Moved generated events to: {args.out_txt}")
