import os
from pathlib import Path

class Settings:
    """
    Centralized configuration for experiments.

    Attributes:
        API_BASE (str): Base URL for the API endpoint.
        API_VERSION (str): Version of the API to use.
        DEPLOYMENT_NAME (str): Name of the deployed model.

        BASE_PATH (Path): Root directory for all project files.
        GUIDELINE_PATH (Path): Path to the guideline JSON file.
        RESULTS_PATH (Path): Directory to store experiment results.
        ANNOTATIONS_PATH (Path): Directory containing human annotation files.

        START_INDEX (int): Index to start processing data.
        END_INDEX (int): Index to end processing data.
        NUM_ITERATIONS (int): Number of iterations for experiment loops.
        MAX_THREADS (int): Maximum number of threads to use for processing.
        TIMEOUT (int): Timeout (in seconds) for API calls.
        MAX_RETRIES (int): Maximum number of retries for failed API calls.
        HUMAN_ANNOTATIONS (int): Total number of human annotation examples.

        CONSTRUCT_BENCHMARK (bool): Whether to construct benchmark datasets.
        BENCHMARK_THRESHOLD (float): Threshold used for benchmark evaluation.
        BENCHMARK_OUTPUT_DIR (Path): Directory to store benchmark results.
    """

    def __init__(self, model_name: str):
        # -------------------
        # API Configuration
        # -------------------
        self.API_BASE = "https://api.example.com"  # anonymized
        self.API_VERSION = "2024-12-01-preview"
        self.DEPLOYMENT_NAME = model_name
        self.SCOPE = "llama_scope"

        # -------------------
        # File Paths
        # -------------------
        self.BASE_PATH = Path("/path/to/project")  # anonymized
        self.GUIDELINE_PATH = self.BASE_PATH / "data" / "guideline.json"
        self.RESULTS_PATH = self.BASE_PATH / "results" / f"experiment_{self.DEPLOYMENT_NAME}"
        self.ANNOTATIONS_PATH = self.BASE_PATH / "data" / "annotations"

        # -------------------
        # Processing Parameters
        # -------------------
        self.START_INDEX = 0
        self.END_INDEX = 122
        self.NUM_ITERATIONS = 10
        self.MAX_THREADS = 30
        self.TIMEOUT = 400
        self.MAX_RETRIES = 3
        self.HUMAN_ANNOTATIONS = 360

        # -------------------
        # Benchmark Settings
        # -------------------
        self.CONSTRUCT_BENCHMARK = True
        self.BENCHMARK_THRESHOLD = 0.9
        self.BENCHMARK_OUTPUT_DIR = self.BASE_PATH / "data" / f"benchmarks/{self.DEPLOYMENT_NAME}"
        os.makedirs(self.BENCHMARK_OUTPUT_DIR, exist_ok=True)
