"""
Contains the code to replicate
the LINGAM experiments.
"""

import os
import time

import fire
import pandas as pd
import ray

from abs_lingam import evaluate_lingam
from learnabs.dataset import generate_datasets
from learnabs.experiments import preprocess_results


def plot(
    results_dir: str = "results/",
    flavor: str = "small",
    plot_dir: str = "plots/",
    store: bool = True,
    show: bool = False,
):
    import matplotlib.pyplot as plt
    import pandas as pd
    import seaborn as sns

    results_fname = os.path.join(results_dir, f"experiment3_{flavor}.csv")
    results = pd.read_csv(results_fname)

    print(f"== Experiment 3 ({flavor}) ==")

    # filter results
    print(f"Total Entries: {len(results)}")
    results = preprocess_results(results)
    print(f"Flavor Entries: {len(results)}")

    for y, y_label in [
        ("eval/abstract_roc_auc", r"ROCAUC $\mathcal{H}$"),
        ("eval/abstract_precision", r"Precision $\mathcal{H}$"),
        ("eval/abstract_recall", r"Recall $\mathcal{H}$"),
        ("eval/abstract_f1", r"F1 $\mathcal{H}$"),
        ("eval/tau_roc_auc", r"ROCAUC $\mathbf{T}$"),
        ("eval/tau_precision", r"Precision $\mathbf{T}$"),
        ("eval/tau_recall", r"Recall $\mathbf{T}$"),
        ("eval/tau_f1", r"F1 $\mathbf{T}$"),
        ("eval/concrete_roc_auc", r"ROCAUC $\mathcal{L}$"),
        ("eval/concrete_precision", r"Precision $\mathcal{L}$"),
        ("eval/concrete_recall", r"Recall $\mathcal{L}$"),
        ("eval/concrete_f1", r"F1 $\mathcal{L}$"),
        ("eval/time", "Time (s)"),
        ("eval/pk_precision", "Prior Knowledge Precision"),
        ("eval/pk_recall", "Prior Knowledge Recall"),
    ]:
        if show:
            plt.figure(figsize=(6, 6))
        else:
            plt.figure(figsize=(6, 3.5))

        sns.lineplot(
            x="dset/cnc_nodes",
            y=y,
            hue="Method",
            style="Method",
            data=results,
        )
        # plt.yscale("log")
        # rename axis
        plt.xlabel(r"Concrete Nodes $|\mathbf{X}|$")
        plt.ylabel(y_label)
        # rename legend
        plt.legend(title="Method", loc="lower right")

        # store plots/exp2_rocauc.pdf
        metric = y.split("/")[1]
        plt.tight_layout()
        if store:
            plt.savefig(os.path.join(plot_dir, f"exp3_{metric}_{flavor}.pdf"))
            plt.savefig(os.path.join(plot_dir, f"exp3_{metric}_{flavor}.pgf"))
        if show:
            plt.show()
        plt.clf()

    if show:
        return results


def run(
    seed: int = 1011329608,
    num_cpus: int = 2,
    num_runs: int = 5,
    flavor: str = "small",
    data_dir: str = "data/",
    results_dir: str = "results/",
    verbose: bool = False,
):
    """run"""

    # initialize ray
    assert num_cpus > 1
    ray.init(num_cpus=num_cpus, num_gpus=0)

    datetime = time.strftime("%Y-%m-%d-%H-%M-%S")
    dset_params = {
        "n_nodes": 5,
        "n_edges": 8,
        "graph_type": "ER",
        "min_readout": 5,
        "max_readout": 10,
        "alpha": 1e3,
        "marginalize_ratio": 0.5,
        "internal": True,
        "n_samples": 50000,
        "noise_term": "exponential",
    }

    range_readout = [(1, 1), (2, 2), (5, 5), (10, 10)]

    if flavor == "small":
        dset_params["n_nodes"] = 5
        dset_params["n_edges"] = 8
    elif flavor == "medium":
        dset_params["n_nodes"] = 10
        dset_params["n_edges"] = 20
    elif flavor == "large":
        dset_params["n_nodes"] = 20
        dset_params["n_edges"] = 40
    else:
        raise ValueError(f"Unknown flavour {flavor}.")

    futures = []
    for min_readout, max_readout in range_readout:
        dset_params["min_readout"] = min_readout
        dset_params["max_readout"] = max_readout
        generate_datasets(dset_params, data_dir, num_runs, force=False)
        n_concrete = 15000
        n_paired = dset_params["n_nodes"] * dset_params["max_readout"] * 2
        for run in range(num_runs):
            # Test the Full Concrete method
            futures.append(
                evaluate_lingam.remote(
                    dset_params.copy(),
                    data_dir,
                    method="Full Concrete",
                    n_paired=0,
                    n_concrete=n_concrete,
                    run=run,
                    shuffle_features=True,
                    normalize=True,
                    seed=seed,
                    verbose=verbose,
                )
            )
            # The the Abs-GT method
            futures.append(
                evaluate_lingam.remote(
                    dset_params.copy(),
                    data_dir,
                    method="Abs-GT",
                    n_paired=0,
                    n_concrete=n_concrete,
                    run=run,
                    shuffle_features=True,
                    normalize=True,
                    seed=seed,
                    verbose=verbose,
                )
            )
            # Test the Abs-Fit method
            futures.append(
                evaluate_lingam.remote(
                    dset_params.copy(),
                    data_dir,
                    method="Abs-Fit",
                    n_paired=n_paired,
                    n_concrete=n_concrete,
                    run=run,
                    shuffle_features=True,
                    normalize=True,
                    seed=seed,
                    verbose=verbose,
                )
            )
            for bootstrap_samples in [1, 2, 5, 10]:
                # Test the Abs-Fit method
                futures.append(
                    evaluate_lingam.remote(
                        dset_params.copy(),
                        data_dir,
                        method="Abs-Fit",
                        n_paired=n_paired,
                        n_concrete=n_concrete,
                        run=run,
                        shuffle_features=True,
                        normalize=True,
                        seed=seed,
                        verbose=verbose,
                        abs_threshold=0.1,
                        bootstrap_samples=bootstrap_samples,
                    )
                )

    # get records and add experiment info
    print(f"Launched {len(futures)} total jobs.")
    records = ray.get(futures)
    print(f"Finished {len(records)} total jobs.")
    for record in records:
        record["experiment/seed"] = seed
        record["experiment/datetime"] = datetime

    # build and append dataframe
    df = pd.DataFrame.from_records(records)
    if not os.path.exists(results_dir):
        os.makedirs(results_dir)
    results_fname = os.path.join(results_dir, f"experiment3_{flavor}.csv")
    df.to_csv(results_fname, index=False)


if __name__ == "__main__":
    fire.Fire()
