# -*- coding: utf-8 -*-
import numpy as np
import matplotlib.pyplot as plt
from sklearn.metrics import auc
from Utils.spot import SPOT
from Utils.metrics import pr_from_events


def calc_point2point(predict, actual):
    """
    calculate f1 score by predict and actual.

    Args:
        predict (np.ndarray): the predict label
        actual (np.ndarray): np.ndarray
    """
    TP = np.sum(predict * actual)
    TN = np.sum((1 - predict) * (1 - actual))
    FP = np.sum(predict * (1 - actual))
    FN = np.sum((1 - predict) * actual)
    precision = TP / (TP + FP + 0.00001)
    recall = TP / (TP + FN + 0.00001)
    f1 = 2 * precision * recall / (precision + recall + 0.00001)
    return f1, precision, recall, TP, TN, FP, FN


def adjust_predicts(score, label,
                    threshold=None,
                    pred=None,
                    calc_latency=False):
    """
    Calculate adjusted predict labels using given `score`, `threshold` (or given `pred`) and `label`.

    Args:
        score (np.ndarray): The anomaly score
        label (np.ndarray): The ground-truth label
        threshold (float): The threshold of anomaly score.
            A point is labeled as "anomaly" if its score is lower than the threshold.
        pred (np.ndarray or None): if not None, adjust `pred` and ignore `score` and `threshold`,
        calc_latency (bool):

    Returns:
        np.ndarray: predict labels
    """
    if len(score) != len(label):
        raise ValueError("score and label must have the same length")
    score = np.asarray(score)
    label = np.asarray(label)
    latency = 0
    if pred is None:
        predict = score < threshold
    else:
        predict = pred
    actual = label > 0.1
    anomaly_state = False
    anomaly_count = 0
    for i in range(len(score)):
        if actual[i] and predict[i] and not anomaly_state:
                anomaly_state = True
                anomaly_count += 1
                for j in range(i, 0, -1):
                    if not actual[j]:
                        break
                    else:
                        if not predict[j]:
                            predict[j] = True
                            latency += 1
        elif not actual[i]:
            anomaly_state = False
        if anomaly_state:
            predict[i] = True
    if calc_latency:
        return predict, latency / (anomaly_count + 1e-4)
    else:
        return predict


def calc_seq(score, label, threshold, calc_latency=False):
    """
    Calculate f1 score for a score sequence
    """
    if calc_latency:
        predict, latency = adjust_predicts(score, label, threshold, calc_latency=calc_latency)
        t = list(calc_point2point(predict, label))
        t.append(latency)
        return t
    else:
        predict = adjust_predicts(score, label, threshold, calc_latency=calc_latency)
        return calc_point2point(predict, label)


def bf_search(score, label, start, end=None, step_num=1, display_freq=1, verbose=True):
    """
    Find the best-f1 score by searching best `threshold` in [`start`, `end`).


    Returns:
        list: list for results
        float: the `threshold` for best-f1
    """
    if step_num is None or end is None:
        end = start
        step_num = 1
    search_step, search_range, search_lower_bound = step_num, end - start, start
    if verbose:
        print("search range: ", search_lower_bound, search_lower_bound + search_range)
    threshold = search_lower_bound
    m = (-1., -1., -1.)
    m_t = 0.0
    for i in range(search_step):
        threshold += search_range / float(search_step)
        target = calc_seq(score, label, threshold, calc_latency=True)
        if target[0] > m[0]:
            m_t = threshold
            m = target
        if verbose and i % display_freq == 0:
            print("cur thr: ", threshold, target, m, m_t)
    print(m, m_t)
    return m, m_t


def pot_eval(init_score, score, label, q=1e-3, level=0.00008):
    """
    Run POT method on given score.
    Args:
        init_score (np.ndarray): The data to get init threshold.
            For `OmniAnomaly`, it should be the anomaly score of train set.
        score (np.ndarray): The data to run POT method.
            For `OmniAnomaly`, it should be the anomaly score of test set.
        label:
        q (float): Detection level (risk)
        level (float): Probability associated with the initial threshold t

    Returns:
        dict: pot result dict
    """
    s = SPOT(q)  # SPOT object
    s.fit(init_score, score)  # data import
    s.initialize(level=level, min_extrema=True)  # initialization step
    ret = s.run(dynamic=False)  # run
    print(len(ret['alarms']))
    print(len(ret['thresholds']))
    pot_th = -np.mean(ret['thresholds'])
    pred, p_latency = adjust_predicts(score, label, pot_th, calc_latency=True)
    p_t = calc_point2point(pred, label)
    print('POT result: ', p_t, pot_th, p_latency)
    return {
        'pot-f1': p_t[0],
        'pot-precision': p_t[1],
        'pot-recall': p_t[2],
        'pot-TP': p_t[3],
        'pot-TN': p_t[4],
        'pot-FP': p_t[5],
        'pot-FN': p_t[6],
        'pot-threshold': pot_th,
        'pot-latency': p_latency
    }

def getMetrics(score,label,threshold):
    pred, p_latency = adjust_predicts(score, label, threshold, calc_latency=True)
    p_t = calc_point2point(pred, label)
    print("f1 score:%f"%p_t[0])

def getPointPosition(label,maxLen=3):
    pNormal=np.zeros(label.shape)
    pPoint=np.zeros(label.shape)
    start=0
    length=0
    for i in range(len(label)):
        if label[i]==0.:
            pNormal[i]=1.
            if start==1 and length<=maxLen:
                pPoint[i-length:i]=1
            start=0
            length=0
        else:
            start=1
            length+=1
    return pPoint,pNormal


def searchThresholdPointandContextual(score,label):
    pPoint,pNormal=getPointPosition(label)
    pCont=np.logical_not(np.logical_or(pPoint,pNormal))
    PointMask=np.logical_or(pPoint,pNormal)
    ContMask=np.logical_or(pCont,pNormal)
    #print(score.shape,label.shape,PointMask.shape)
    pointRes=searchThreshold(score[PointMask],label[PointMask])
    contRes=searchThreshold(score[ContMask],label[ContMask])
    return pointRes,contRes


def transformFormat4Affliation(series):
    start=0
    length=0
    events=[]
    for i in range(len(series)):
        if series[i]==0 or i==len(series)-1:
            if start==1:
                events.append((i-length,i))
            start=0
            length=0
        if series[i]==1:
            start=1
            length+=1
    return events

def affliationMetrics(score,label):
    mins = np.min(score)
    maxs = np.max(score)
    stride = (maxs - mins) / 1000
    threshold = -1
    maxF1 = [0] * 7
    i = mins + stride
    precisions = []
    recalls = []
    label=label!=0
    while i < maxs + stride:
        pred, p_latency = adjust_predicts(score, label, i, calc_latency=True)
        p_t = calc_point2point(pred, label)
        precisions.append(p_t[1])
        recalls.append(p_t[2])
        if p_t[0] > maxF1[0]:
            maxF1 = p_t
            threshold = i
        i += stride
    score = np.array(score)
    pred = score < threshold
    predForm=transformFormat4Affliation(pred)
    labelForm=transformFormat4Affliation(label)
    if len(labelForm)==0:
        affiRes=dict({'precision': 0,
                     'recall': 0,
                     'individual_precision_probabilities': 0,
                     'individual_recall_probabilities': 0,
                     'individual_precision_distances': 0,
                     'individual_recall_distances': 0})
    else:
        affiRes=pr_from_events(predForm,labelForm,(0,len(pred)))
    return affiRes

def searchThreshold(score,label):
    mins=np.min(score)
    maxs=np.max(score)
    stride=(maxs-mins)/1000
    threshold=-1
    maxF1=[0]*7
    i=mins+stride
    precisions=[]
    recalls=[]
    while i<maxs+stride:
        pred, p_latency = adjust_predicts(score, label, i, calc_latency=True)
        p_t = calc_point2point(pred, label)
        precisions.append(p_t[1])
        recalls.append(p_t[2])
        if p_t[0]>maxF1[0]:
            maxF1=p_t
            threshold=i
        i+=stride
    score=np.array(score)
    mask=score<threshold
    num=mask.sum()
    ratio=num/score.size
    AUPR=0.
    sumv=[]
    for i in range(1,len(recalls)):
        AUPR+=(recalls[i]-recalls[i-1])*precisions[i-1]
        sumv.append((recalls[i]-recalls[i-1])*precisions[i-1])
    return {
        'pot-f1': maxF1[0],
        'pot-precision': maxF1[1],
        'pot-recall': maxF1[2],
        'pot-TP': maxF1[3],
        'pot-TN': maxF1[4],
        'pot-FP': maxF1[5],
        'pot-FN': maxF1[6],
        'pot-threshold': threshold,
        'threshold-ratio':ratio,
        "AUPR":AUPR
    }

def getF1score(score,label,threshold):
    pred, p_latency = adjust_predicts(score, label, threshold, calc_latency=True)
    p_t = calc_point2point(pred, label)
    return {
        'pot-f1': p_t[0],
        'pot-precision': p_t[1],
        'pot-recall': p_t[2],
        'pot-TP': p_t[3],
        'pot-TN': p_t[4],
        'pot-FP': p_t[5],
        'pot-FN': p_t[6],
        'pot-threshold': threshold,
        'pot-latency': p_latency
    }