import torch
from torch.nn import Linear, Parameter
import torch.nn.functional as F
import torch.nn as nn
import torch_geometric
from torch_geometric.nn import MessagePassing
from torch_geometric.utils import add_self_loops, degree
from hydra.utils import instantiate

#TODO(tm): send in the device while creating the model. 
device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')



class LinearMultiplyIsingMessagePassingLayer(MessagePassing):
    """
        # x = ReLU(W2 * message + b_2) * tanh(W_1 * message + b_1)
        v2: 
            x = (W2 * message + b_2) * tanh(W_1 * message + b_1)
    """
    def __init__(self, in_channels=1, out_channels=1, 
                 cached=False, normalize=False, Jst=1, 
                 add_bias=True, initialize_w_zero=False):
        super().__init__(aggr='add')
        self.tanh_activation = nn.Tanh()
        self.Jst = Jst
        self.J1 = torch.nn.Linear(in_channels, out_channels)
        self.J2 = torch.nn.Linear(in_channels, out_channels)
        if initialize_w_zero:
            self.J1.weight = nn.Parameter(
                torch.zeros_like(self.J1.weight), requires_grad=True)
            self.J1.bias = nn.Parameter(
                torch.zeros_like(self.J1.bias), requires_grad=True)
            self.J2.weight = nn.Parameter(
                torch.zeros_like(self.J2.weight), requires_grad=True)
            self.J2.bias = nn.Parameter(
                torch.zeros_like(self.J2.bias), requires_grad=True)
    
    def reset_parameters(self):
        self.J1.reset_parameters()
        self.J1.bias.zero_()
        self.J2.reset_parameters()
        self.J2.bias.zero_()

    # def forward(self, x, edge_index, edge_potential, edge_weight=None):
    def forward(self, message, edge_index, 
                node_potential, edgewise_edge_index=None):
        # NOTE: we can't really add any edge index to the mix. this will mess things up for us.
        # edge_index, _ = add_self_loops(edge_index, num_nodes=x.size(0))

        middle1 = self.J1(message).tanh()
        # middle = torch.einsum('h l, b l -> b h', 
        #                       self.J1.weight.tanh(), message.tanh())
        # middle = middle + self.J1.bias
        middle2 = self.J2(message)
        middle = torch.mul(middle1, middle2)
        
        # middle = torch.atanh(self.J1(message))
        
        # Step 3: Compute normalization.
        row, col = edgewise_edge_index
        deg = degree(col, message.size(0), dtype=message.dtype)
        deg_inv_sqrt = deg.pow(-0.5)
        deg_inv_sqrt[deg_inv_sqrt == float('inf')] = 0
        norm = deg_inv_sqrt[row] * deg_inv_sqrt[col]
        norm = torch.ones_like(norm)

        # Step 4-5: Start propagating messages.
        out = self.propagate(edgewise_edge_index, x=middle, norm=norm) 
        
        x = out + node_potential[edge_index[0]]
        x = self.tanh_activation(x)
        return x
    

class BeliefPropLayersImprovedMessagePassing(torch.nn.Module):
    def __init__(self, 
                 layer_class,
                 num_hidden_layers=0, 
                 in_channels=1, 
                 hidden_channels=1, 
                 out_channels=1,
                 add_bias=True,
                 initialize_w_zero=False,
                 Jst=1,
                 message_aggregate='learned', 
                 **kwargs):
        super().__init__()
        self.num_hidden_layers = num_hidden_layers
        self.Jst = Jst
        self.add_bias = add_bias
        self.initialize_w_zero = initialize_w_zero
        print(self.Jst)
        assert message_aggregate in ['fixed', 'learned'], "invalid message aggregate"
        self.message_aggregate = message_aggregate
        self.layer_class = layer_class
        self.conv1 = LinearMultiplyIsingMessagePassingLayer(
            in_channels=in_channels, 
            out_channels=hidden_channels,
            add_bias=self.add_bias,
            initialize_w_zero=initialize_w_zero,
            cached=True, 
            Jst=self.Jst,
            normalize=False)
        
        self.hidden_layers = nn.ModuleList()
        
        for _ in range(self.num_hidden_layers):
            self.hidden_layers.append(LinearMultiplyIsingMessagePassingLayer(
                        in_channels=hidden_channels, 
                        out_channels=hidden_channels,
                        add_bias=self.add_bias,
                        initialize_w_zero=initialize_w_zero,
                        cached=True, 
                        Jst=self.Jst,
                        normalize=False))

        self.conv2 = LinearMultiplyIsingMessagePassingLayer(
            in_channels=hidden_channels, 
            out_channels=out_channels,
            add_bias=self.add_bias,
            initialize_w_zero=initialize_w_zero,
            cached=True, 
            Jst=self.Jst,
            normalize=False)
        if self.message_aggregate == 'learned':
            self.aggregation_weight = nn.Linear(
                out_channels, out_channels, bias=False)
            self.aggregation_weight.weight = nn.Parameter(
                torch.zeros_like(self.aggregation_weight.weight),
                requires_grad=True
            )

    def final_expectation(self, edge_messages, edge_index, node_potentials):
        aggr = torch_geometric.nn.aggr.SumAggregation()
        # em = torch.atanh(torch.tanh(torch.tensor([Jst]).to(device) * edge_messages))
        #NOTE: maybe this needs a learnable parameter as well.
        if self.message_aggregate == 'fixed':
            em = torch.atanh(
                torch.tanh(torch.tensor([self.Jst]).to(device) * edge_messages))
        elif self.message_aggregate == 'learned':
            em = torch.tanh(self.aggregation_weight(edge_messages))
        sum_of_neighbors = aggr(em, edge_index[1])
        final = torch.tanh(node_potentials + sum_of_neighbors)
        return final
        
    def forward(self, x, data, edgewise_edge_index):
        # ==== data from the main graph ==== 
        node_feature = data.x.type(torch.FloatTensor).to(device)
        edge_index = data.edge_index

        x = self.conv1(
            x, edge_index, node_feature, 
            edgewise_edge_index=edgewise_edge_index)
        for i in range(self.num_hidden_layers):
            x =  self.hidden_layers[i](
                x, edge_index, node_feature, 
                edgewise_edge_index=edgewise_edge_index)
        x = self.conv2(x, edge_index, node_feature, edgewise_edge_index=edgewise_edge_index)
        final_x = self.final_expectation(x, edge_index, node_feature)
        return final_x