import torch
from torch.nn import Linear, Parameter
import torch.nn.functional as F
import torch.nn as nn
import torch_geometric
from torch_geometric.nn import MessagePassing, GCNConv
from torch_geometric.utils import add_self_loops, degree

#TODO(tm): send in the device while creating the model. 
device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')

class UpDownLeaveNodeDirectionality(MessagePassing):
    def __init__(self, 
                 in_channels=1, 
                 out_channels=1, 
                 Jst=1, 
                 add_bias=True, 
                 undirected_edgewise_edge_index=False,
                 initialize_w_zero=False):
        super().__init__(aggr='add')
        self.tanh_activation = nn.Tanh()
        self.Jst = Jst
        self.J_source = torch.nn.Linear(in_channels, out_channels)
        self.J_target = torch.nn.Linear(in_channels, out_channels)
        self.undirected_edgewise_edge_index = undirected_edgewise_edge_index
        self.aggr = torch_geometric.nn.aggr.MeanAggregation()
        if initialize_w_zero:
            self.J_source.weight = nn.Parameter(
                torch.zeros_like(self.J_source.weight), requires_grad=True)
            self.J_source.bias = nn.Parameter(
                torch.zeros_like(self.J_source.bias), requires_grad=True)
            self.J_target.weight = nn.Parameter(
                torch.zeros_like(self.J_target.weight), requires_grad=True)
            self.J_target.bias = nn.Parameter(
                torch.zeros_like(self.J_target.bias), requires_grad=True)
    
    def reset_parameters(self):
        self.J_source.reset_parameters()
        self.J_source.bias.zero_()
        self.J_target.reset_parameters()
        self.J_target.bias.zero_()
    
    def get_edge_direction_mask(self, edge_index):
        indices_up = torch.where(edge_index[0, :] < edge_index[1,:])[0]
        mask = torch.zeros_like(edge_index[0])
        mask[indices_up] = 1
        return mask


    def get_norm(self, edge_index, feature, node_feature=None):
        row, col = edge_index 
        deg = degree(col, feature.size(0), dtype=feature.dtype)
        deg_inv_sqrt = deg.pow(-0.5)
        deg_inv_sqrt[deg_inv_sqrt == float('inf')] = 0
        norm = deg_inv_sqrt[row] * deg_inv_sqrt[col]
        return norm

    def forward(self, x_up, x_down, edge_index, edgewise_edge_index=None):
        x_up_in = x_up
        x_down_in = x_down
        m_up = self.J_source(x_up) 
        m_down = self.J_target(x_down)

        edge_dir_mask = self.get_edge_direction_mask(edge_index)

        message_up = (m_up[edge_index[0]] * edge_dir_mask.unsqueeze(-1) 
                   + m_down[edge_index[1]] * (1 - edge_dir_mask).unsqueeze(-1))

        message_down = (m_up[edge_index[0]] * (1 - edge_dir_mask.unsqueeze(-1))
                   + m_down[edge_index[1]] * edge_dir_mask.unsqueeze(-1))

        norm = self.get_norm(edgewise_edge_index, message_up)

        out_up = self.propagate(edgewise_edge_index, x=message_up, norm=norm) 
        out_down = self.propagate(edgewise_edge_index, x=message_down, norm=norm) 

        x_up = self.aggr(out_up, edge_index[0])
        x_down = self.aggr(out_down, edge_index[1])

        x_up = self.tanh_activation(x_up + x_up_in)
        x_down = self.tanh_activation(x_down + x_down_in)
        return x_up, x_down


class NodeBeliefPropModel(torch.nn.Module):
    def __init__(self, 
                 layer_class,
                 num_hidden_layers=0, 
                 in_channels=1, 
                 hidden_channels=1, 
                 out_channels=1,
                 add_bias=True,
                 undirected_edgewise_edge_index=False,
                 initialize_w_zero=False,
                 Jst=1,
                 message_aggregate='learned', 
                 **kwargs):
        super().__init__()
        self.num_hidden_layers = num_hidden_layers
        self.Jst = Jst
        self.add_bias = add_bias
        self.initialize_w_zero = initialize_w_zero
        self.undirected_edgewise_edge_index = undirected_edgewise_edge_index
        assert message_aggregate in ['fixed', 'learned'], "invalid message aggregate"
        self.message_aggregate = message_aggregate
        self.layer_class = UpDownLeaveNodeDirectionality
        self.conv_block1 = self.layer_class(
            in_channels=in_channels, 
            out_channels=hidden_channels,
            undirected_edgewise_edge_index=undirected_edgewise_edge_index,
            add_bias=self.add_bias,
            initialize_w_zero=initialize_w_zero,)
        
        hidden_layers = nn.ModuleList()
        
        for _ in range(self.num_hidden_layers):
            hidden_layers.append(
                self.layer_class(
                    hidden_channels, 
                    hidden_channels,
                    undirected_edgewise_edge_index=undirected_edgewise_edge_index,
                    add_bias=self.add_bias,
                    initialize_w_zero=initialize_w_zero,)
            )
        self.hidden_layers = hidden_layers
        self.conv_block2 = self.layer_class(
            in_channels=hidden_channels, 
            out_channels=hidden_channels,
            undirected_edgewise_edge_index=undirected_edgewise_edge_index,
            add_bias=self.add_bias,
            initialize_w_zero=initialize_w_zero,)

        if self.message_aggregate == 'learned':
            self.aggregation_weight = nn.Linear(
                hidden_channels, out_channels, bias=False)
            self.aggregation_weight.weight = nn.Parameter(
                torch.zeros_like(self.aggregation_weight.weight),
                requires_grad=True
            )
        self.aggr = torch_geometric.nn.aggr.SumAggregation()

    def final_expectation(self, x_up, x_down, edge_index, node_potentials=None):
        edge_messages = x_up[edge_index[0]] + x_down[edge_index[1]]
        aggr = torch_geometric.nn.aggr.SumAggregation()
        if self.message_aggregate == 'fixed':
            em = torch.atanh(
                torch.tanh(torch.tensor([self.Jst]).to(device) * edge_messages))
        elif self.message_aggregate == 'learned':
            em = torch.tanh(self.aggregation_weight(edge_messages))
        sum_of_neighbors = aggr(em, edge_index[1])
        if node_potentials is not None:
            final = torch.tanh(node_potentials + sum_of_neighbors)
        else:
            final = torch.tanh(sum_of_neighbors)
        return final

    def forward(self, data, x=None, edgewise_edge_index=None):
        # ==== data from the main graph ==== 
        x = data.x.type(torch.FloatTensor).to(device)
        edge_index = data.edge_index
        # edgewise_edge_index = data.edge_index
        x_up, x_down = self.conv_block1(
            x_up=x, 
            x_down=x, 
            edge_index=edge_index, 
            edgewise_edge_index=edgewise_edge_index)

        for i in range(self.num_hidden_layers):
            x_up_in = x_up
            x_down_in = x_down
            x_up, x_down = self.hidden_layers[i](
                x_up=x_up, 
                x_down=x_down, 
                edge_index=edge_index,
                edgewise_edge_index=edgewise_edge_index)
            # x_up = x_up + x_up_in
            # x_down = x_down + x_down_in 

        x_up, x_down = self.conv_block2(
            x_up=x_up, 
            x_down=x_down, 
            edge_index=edge_index,
            edgewise_edge_index=edgewise_edge_index)
        final_x = self.final_expectation(x_up, x_down, edge_index, node_potentials=x)
        return final_x