import torch
from torch.nn import Linear, Parameter
import torch.nn.functional as F
import torch.nn as nn
import torch_geometric
from torch_geometric.nn import MessagePassing, GCNConv
from torch_geometric.utils import add_self_loops, degree

#TODO(tm): send in the device while creating the model. 
device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')

class UpDownLeaveNodeDirectionality(MessagePassing):
    def __init__(self, 
                 in_channels=1, 
                 out_channels=1, 
                 Jst=1, 
                 add_bias=True, 
                 undirected_edgewise_edge_index=False,
                 initialize_w_zero=False):
        super().__init__(aggr='add')
        self.tanh_activation = nn.Tanh()
        self.Jst = Jst
        self.J_source = torch.nn.Linear(in_channels, out_channels)
        self.undirected_edgewise_edge_index = undirected_edgewise_edge_index
        if initialize_w_zero:
            self.J_source.weight = nn.Parameter(
                torch.zeros_like(self.J_source.weight), requires_grad=True)
            self.J_source.bias = nn.Parameter(
                torch.zeros_like(self.J_source.bias), requires_grad=True)
    
    def reset_parameters(self):
        self.J_source.reset_parameters()
        self.J_source.bias.zero_()
        self.J_target.reset_parameters()
        self.J_target.bias.zero_()
    

    def get_norm(self, edge_index, feature):
        row, col = edge_index 
        deg = degree(col, feature.size(0), dtype=feature.dtype)
        deg_inv_sqrt = deg.pow(-0.5)
        deg_inv_sqrt[deg_inv_sqrt == float('inf')] = 0
        norm = deg_inv_sqrt[row] * deg_inv_sqrt[col]
        return norm

    def forward(self, x_up, edge_index):
        m_up = self.J_source(x_up) 

        norm = self.get_norm(edge_index, m_up)

        out_up = self.propagate(edge_index, x=m_up, norm=norm) 

        x_up = self.tanh_activation(out_up) 
        return x_up


class NodeBeliefPropModel(torch.nn.Module):
    def __init__(self, 
                 layer_class,
                 num_hidden_layers=0, 
                 in_channels=1, 
                 hidden_channels=1, 
                 out_channels=1,
                 add_bias=False,
                 undirected_edgewise_edge_index=False,
                 initialize_w_zero=False,
                 Jst=1,
                 message_aggregate='learned', 
                 **kwargs):
        super().__init__()
        self.num_hidden_layers = num_hidden_layers
        self.Jst = Jst
        self.add_bias = add_bias
        self.initialize_w_zero = initialize_w_zero
        self.undirected_edgewise_edge_index = undirected_edgewise_edge_index
        assert message_aggregate in ['fixed', 'learned'], "invalid message aggregate"
        self.message_aggregate = message_aggregate
        self.layer_class = UpDownLeaveNodeDirectionality
        self.add_bias = False
        self.conv_block1 = self.layer_class(
            in_channels=in_channels, 
            out_channels=hidden_channels,
            undirected_edgewise_edge_index=undirected_edgewise_edge_index,
            add_bias=self.add_bias,
            initialize_w_zero=initialize_w_zero,)
        
        hidden_layers = nn.ModuleList()
        
        for _ in range(self.num_hidden_layers):
            hidden_layers.append(
                self.layer_class(
                    hidden_channels, 
                    hidden_channels,
                    undirected_edgewise_edge_index=undirected_edgewise_edge_index,
                    add_bias=self.add_bias,
                    initialize_w_zero=initialize_w_zero,)
            )
        self.hidden_layers = hidden_layers
        self.conv_block2 = self.layer_class(
            in_channels=hidden_channels, 
            out_channels=hidden_channels,
            undirected_edgewise_edge_index=undirected_edgewise_edge_index,
            add_bias=self.add_bias,
            initialize_w_zero=initialize_w_zero,)

        self.aggregation_weight = nn.Linear(
            hidden_channels, out_channels)

    def forward(self, data):
        # ==== data from the main graph ==== 
        x = data.x.type(torch.FloatTensor).to(device)
        edge_index = data.edge_index
        x_up = self.conv_block1(
            x_up=x, 
            edge_index=edge_index)

        for i in range(self.num_hidden_layers):
            x_in = x_up
            x_up = self.hidden_layers[i](
                x_up=x_up, 
                edge_index=edge_index) + x_in

        # x_up = self.conv_block2(
        #     x_up=x_up, 
        #     edge_index=edge_index)

        final_x = torch.tanh(self.aggregation_weight(x_up))
        return final_x