#!/usr/bin/env python
# coding: utf-8

# In[15]:


import numpy as np
import pandas as pd
from matplotlib import pyplot as plt
#from convert2WFC import transformation
import itertools
import math
import time
import os
import sys
#from find_max import find_max

np.random.seed(42)


# In[2]:


lattice = np.load('../spin_configs/lattice_32x32.npy')


# In[3]:


Nx, Ny = lattice.shape


# In[4]:


#Nx,Ny = lattice_n.shape

#plt.figure()
#plt.colorbar(plt.imshow(lattice,cmap = 'gray'))

#plt.grid(False)
#plt.show()


# In[24]:


def logarithmic_cooling_fixed_range(t, total_steps, T_start, T_end, scale):
    """
    Logarithmic cooling schedule scaled to a fixed temperature range.
    
    Args:
        t (int): The current time step or iteration number.
        total_steps (int): The total number of steps in the annealing process.
        T_start (float): The initial temperature.
        T_end (float): The final temperature.
        
    Returns:
        float: The temperature at step t.
    """
    if t >= total_steps:
        return T_end
    
    # Normalized progress (0 to 1) based on a logarithmic scale
    log_progress = math.log(1 + scale*t) / math.log(1 + scale*total_steps)
    
    # Linear interpolation between T_start and T_end
    # This formula maps the logarithmic curve to the desired temperature range.
    return T_start - (T_start - T_end) * log_progress


# In[25]:


def compute_curie_weiss_hamiltonian(spins, noise_level):
    """
    Compute noisy energy from a 2D spin configuration using Fourier method.

    Parameters
    ----------
    spins : 2D array-like
        Array of ±1 spin values.
    """
    
    spins = np.asarray(spins)
    M, K = spins.shape
    N = M * K

    # Phase: +1 → 0.0, -1 → π
    phases = np.where(spins == 1, 0.0, np.pi)
    cwf = np.exp(1j * phases)

    # 2D FFT with zero-frequency shift
    fft_res = np.fft.fftshift(np.fft.fft2(cwf))
    dc = np.abs(fft_res[M // 2, K // 2]) ** 2

    base_E = -(dc - N) / (2 * N)  #-dc / N

    # Add proportional noise
    if noise_level > 0:
        sigma = noise_level * abs(base_E)
        epsilon = np.random.normal(0.0, sigma)
        noisy_E = base_E + epsilon
    else:
        noisy_E = base_E
    
    return float(noisy_E)


# In[26]:


def metropolis(spin_arr, times, T_start, T_end, energy, noise, num_flips, numexpt):
    
    spin_arr = spin_arr.copy()
    net_spins = np.zeros(times-1, dtype=np.float64)
    curr_BJ = np.zeros(times-1, dtype=np.float64)
    net_energy = np.zeros(times-1, dtype=np.float64)
    delta_E=np.zeros(times-1, dtype=np.float64)
    exec_time = np.zeros([times-1, 7])
    
    initial_energies = []
    for _ in range(numexpt):
        initial_energies.append(compute_curie_weiss_hamiltonian(spin_arr, noise))
    initial_energy = np.mean(initial_energies)
    #initial_energy = compute_curie_weiss_hamiltonian(spin_arr, noise)
    
    time = 0  # Manual time counter

    while time < times - 1:

        curr_beta = 1 / (logarithmic_cooling_fixed_range(time, times, T_start, T_end, 1))
        #print(time, curr_beta)
        curr_BJ[time] = curr_beta
        
        x = np.random.randint(0, spin_arr.shape[0], num_flips)
        y = np.random.randint(0, spin_arr.shape[1], num_flips)
        #print(x,y)

        spin_initial = spin_arr.copy()
        spin_flip = spin_arr.copy()
        spin_flip[x, y] = -spin_arr[x, y]

        flip_energies = []
        for _ in range(numexpt):
            flip_energies.append(compute_curie_weiss_hamiltonian(spin_flip, noise))
        flip_energy = np.mean(flip_energies)
        
        
        dE = flip_energy - initial_energy

        #print(f"Time step: {time}, initial_energy={initial_energy}")
        #dE = dE/num_flips
        
        if dE <= 0:
            spin_arr = spin_flip
            initial_energy = initial_energy + dE
            #energy += dE
            
        elif (dE>0) and (np.random.random() < np.exp(-curr_beta*dE)):
            spin_arr = spin_flip
            initial_energy = initial_energy + dE
            #energy += dE
            
        #print(f"Time step: {time}, final_energy={energy}")
        net_spins[time] = spin_arr.sum()
        net_energy[time] = initial_energy
        delta_E[time] = dE
        
        if (time%20000==0):
            print(f"Time step: {time}, curr_BJ={curr_beta}, dE={dE}, avg. spin={net_spins[time]/((spin_arr.shape[0])*(spin_arr.shape[1]))}")

        time += 1  # ✅ Now time advances per MC step
        #print(time, energy)

    return net_spins, net_energy, spin_arr, delta_E, curr_BJ


# # Run Experiments

# In[29]:


# Define parameter grids
times = 400000
noise_levels = [0.15,0.2]
numexpt_values = [1]
num_flips = 1

#Define beta pairs (start_beta, end_beta)
BJ_pairs = [(0.5, 0.5), (0.7, 0.7), (0.85, 0.85), (1, 1), (1.015,1.015), (1.03, 1.03), (1.06, 1.06), (1.09,1.09), (1.12,1.12), (1.15,1.15),
            (1.18,1.18), (1.21,1.21), (1.24,1.24), (1.27,1.27), (1.3,1.3), (1.36,1.36), (1.4, 1.4), (1.5, 1.5), (1.65, 1.65), (1.8, 1.8),
            (2.0, 2.0), (2.2, 2.2), (2.6, 2.6), (3.0, 3.0) ]

#BJ_pairs = [(3.0, 3.0) ]

# Convert to temperature pairs using T = 1 / β
temperature_pairs = [(1 / BJ_start, 1 / BJ_end) for BJ_start, BJ_end in BJ_pairs]

# Loop over all combinations
for noise, numexpt, (T_initial, T_final) in itertools.product(noise_levels, numexpt_values, temperature_pairs):
    
    key = f"noise={noise}_numexpt={numexpt}_Tinit={T_initial:.3f}_Tfinal={T_final:.3f}"
    print(f"Running: {key}")

    
    # Run the Metropolis simulation
    spins, energies, final_lattice, delta_E, BJ = metropolis(lattice.copy(), times, T_initial, T_final,
        compute_curie_weiss_hamiltonian(lattice, noise),
        noise, num_flips, numexpt)

    # Ensure the folder exists
    folder = (f"quenching/{Nx}_by_{Ny}/noise{noise}/numexpt{numexpt}/")
    os.makedirs(folder, exist_ok=True)
    
    # Construct filename
    filename = f"SA_{Nx}x{Ny}_noise{noise}_numexpt{numexpt}_beta{(1/T_initial):.3f}.csv"

    # Full path including folder
    filepath = os.path.join(folder, filename)

    # Prepare data
    steps = np.arange(times - 1)
    beta_values = np.array(BJ)
    magnetization = np.array(spins) / (Nx * Ny)

    # Stack columns: step, beta, magnetization
    history = np.column_stack((steps, beta_values, magnetization))

    # Save to CSV
    np.savetxt(filepath, history, delimiter=",", header="Step,Beta,Magnetization", comments='')


# In[31]:


print ('End of SA')


# In[ ]:




