from abc import ABC, abstractmethod
from typing import List, Optional, Dict, Callable, Any, Tuple

import logging
import inspect
import json5
from colorama import Fore
from dataclasses import dataclass
from transformers import AutoTokenizer

from organisation.env.config import (
    TOOL_ACTOR_ID,
    INCENTIVES_ENABLED,
    LLM_USAGE_METRICS,
    QWEN_MODEL,
)
from ..tasks import Task, TaskStatus
from ..messages import Message
from ..memory import SummarizingMemory
from ..llm_client import get_llm_chat
from ..tool_registry import get_actor_tools
from ..incentives import generate_random_incentive
from .utils import ToolCall, SimTimeAdapter, _annotation_to_schema
from .meeting import Meeting


import simpy


logger = logging.getLogger(__name__)

logger.setLevel(logging.INFO)

# Reuse one logger object (cheap).
_LLM_USAGE_LOGGER = logging.getLogger("LLM.usage")


@dataclass
class _LLMUsageCounters:
    prompt: int = 0
    completion: int = 0
    calls: int = 0


# ————————————————————————————————————————————————————————————
# Actor base class
# ————————————————————————————————————————————————————————————


role_color = {
    "Investigator": Fore.RED,
    "Legal Team": Fore.GREEN,
    "Sponsor": Fore.BLUE,
    "Statistician": Fore.MAGENTA,
    "Regulatory Agency": Fore.CYAN,
}


class Actor(ABC):
    """
    Base class for all actors in the simulation.
    Each actor has a unique ID, a memory, and can perform actions.
    """

    _registry: Dict[int, "Actor"] = {}  # Global registry of actor instances

    def __init__(
        self,
        env: simpy.Environment,
        simulation: Any,
        actor_id: int,
        llm_client: Any,
        llm_kwargs: Optional[Dict[str, Any]] = None,
        incentives_enabled: Optional[bool] = None,
        incentive_text: Optional[str] = None,
        memory: Optional[SummarizingMemory] = None,
        task_dictionary: Optional[List[Dict[str, Any]]] = None,
    ):
        self.env = env
        self.simulation = simulation
        self.drug = simulation.drug
        self.actor_id = actor_id
        self.llm_client = llm_client
        self.llm_kwargs = llm_kwargs or {}
        self.active_meeting = None
        self.tasks: dict[str, Task] = {}
        self.activity = None
        for entry in task_dictionary or []:
            t = Task(
                _id=entry["task_id"],
                name=entry["task_name"],
                description=entry.get("description", ""),
                relies_on=entry.get("relies_on", []),
                mandatory=bool(entry.get("mandatory", False)),
                phase=entry.get("phase", "A"),
            )
            self.tasks[t.name] = t

        # per-actor override beats global
        self._incentives_enabled = (
            INCENTIVES_ENABLED
            if incentives_enabled is None
            else bool(incentives_enabled)
        )

        # Precompute text once (cheap; pure function now that incentives are hard-coded)
        try:
            self.incentive_text = (
                generate_random_incentive(self.org_role)
                if self._incentives_enabled
                else ""
            )
            if self._incentives_enabled and self.incentive_text:
                logging.getLogger("incentives").debug(
                    f"Actor[{self.actor_id}:{self.org_role}] incentive: {self.incentive_text}"
                )
        except Exception:
            self.incentive_text = ""  # never break actor init

        # Initialize attention resource for meetings
        self.attention = simpy.Resource(env, capacity=1)

        # register
        if actor_id in Actor._registry:
            raise ValueError(f"Duplicate actor_id: {actor_id}")
        Actor._registry[actor_id] = self

        self.memory = memory or SummarizingMemory(actor=self)
        self.state = "Inactive"
        self.messages_sent = 0
        self.tokens_produced = 0
        self._llm_usage = _LLMUsageCounters()

        # logger
        base = logging.getLogger(f"Actor[{actor_id}:{self.__class__.__name__}]")
        self.logger = SimTimeAdapter(base, self.env)

        self.tokenizer = AutoTokenizer.from_pretrained(QWEN_MODEL)

    # Convenience for composing system prompts
    def incentive_clause(self) -> str:
        return (
            "\n" + self.incentive_text
            if (self._incentives_enabled and self.incentive_text)
            else ""
        )

    def _log(self, level: int, msg: str, *args, **kwargs):
        """
        Log a message at given level, injecting the current sim time.
        """
        extra = {"sim_time": self.env.now}
        self.logger.log(level, msg, *args, extra=extra, **kwargs)

    def info(self, msg: str, *args, **kwargs):
        self._log(logging.INFO, msg, *args, **kwargs)

    def debug(self, msg: str, *args, **kwargs):
        self._log(logging.DEBUG, msg, *args, **kwargs)

    def get_openai_tools(self) -> list[dict]:
        """Return OpenAI 'tools' (tools-first API)."""
        tools = []
        for name, (fn, description) in get_actor_tools(self).items():
            sig = inspect.signature(fn)
            properties = {}
            required = []

            for p in sig.parameters.values():
                if p.kind in (p.VAR_POSITIONAL, p.VAR_KEYWORD):
                    # skip *args/**kwargs from schema
                    continue
                schema = _annotation_to_schema(p.annotation)
                # keep descriptions simple; you can enrich from docstrings if you want
                properties[p.name] = {**schema, "description": p.name}
                if p.default is inspect._empty:
                    required.append(p.name)

            params = {"type": "object", "properties": properties}
            if required:
                params["required"] = required

            tools.append(
                {
                    "type": "function",
                    "function": {
                        "name": name,
                        "description": description or "",
                        "parameters": params,
                    },
                }
            )
        return tools

    @property
    def tools(self) -> Dict[str, Tuple[Callable[..., Any], str]]:
        # Gather only those methods decorated with @actor_tool
        return get_actor_tools(self)

    @classmethod
    def get_by_id(cls, actor_id: int) -> Optional["Actor"]:
        """
        Retrieve an Actor instance by its ID.
        Returns None if the ID is not found.
        """
        return cls._registry.get(actor_id)

    def get_task_name_from_id(self, task_id: int) -> Optional[str]:
        """Return the task name for a given ID if this actor knows it; else None."""
        for task in self.tasks.values():
            if task._id == task_id:
                return task.name
        return None

    def get_memory(self) -> SummarizingMemory:
        return self.memory

    def get_all_messages(self) -> list[Message]:
        return list(self.memory._all_messages)

    def receive_message(self, message: Message):
        """
        Receive a message and process it.
        """
        # log every incoming message at DEBUG
        self.debug(
            "receive_message \u2190 from=%r to=%r content=%r",
            message.sender,
            message.recipient,
            message.content,
        )
        self.memory.add(message)

        # Track messages the actor itself produced (by ID)
        if message.sender == self.actor_id:
            self.messages_sent += 1
            # message.content may be None when function_call-only
            text = message.content or ""
            self.tokens_produced += len(text.split())

    def get_attention(self, duration, activity_name):
        """
        Request attention for a specific duration and activity.
        """
        self.activity = (activity_name, duration)
        with self.attention.request() as req:
            yield req
            for i in range(duration):
                yield self.env.timeout(1)
                self.activity = (activity_name, duration - i - 1)
        self.activity = None

    def _hold_attention_until_event(self, event, activity_name: str):
        """Reserve attention until `event` fires; non-blocking scheduler."""

        def _runner():
            self.info(f"🔒 Attention reserved for '{activity_name}'")
            with self.attention.request() as req:
                self.activity = (activity_name, None)
                # attach a debug label to the Request itself
                setattr(req, "label", activity_name)
                yield req
                setattr(self, "_attention_since", self.env.now)
                try:
                    yield event
                finally:
                    self.activity = None
            self.info(f"🔓 Attention released for '{activity_name}'")

        self.env.process(_runner())

    # ---- helpers to detect running/queued studies for this actor ----
    def active_study_ids(self) -> list[str]:
        """IDs of studies in env.studies that are assigned to this actor and not completed yet."""
        studies = getattr(self.env, "studies", {}) or {}
        out = []
        for s in studies.values():
            try:
                if getattr(s, "responsible", None) is self and not getattr(
                    s, "completed", False
                ):
                    out.append(getattr(s, "study_id", ""))
            except Exception:
                continue
        return out

    def has_active_study(self) -> bool:
        return len(self.active_study_ids()) > 0

    # LLM Usage logging helpers
    def _estimate_tokens(self, text: str) -> int:
        """
        Fallback token estimator using the actor's tokenizer if available,
        else a rough chars->tokens heuristic.
        """
        try:
            if hasattr(self, "tokenizer") and self.tokenizer is not None:
                return len(self.tokenizer.encode(text or ""))
        except Exception:
            pass
        # very rough fallback
        return max(0, len(text or "") // 4)

    def _log_llm_usage(self, messages, response, context: str) -> None:
        """
        Accumulate token usage for this actor. We do not log per-call; totals are emitted later.
        Uses API-reported usage when available, otherwise a very cheap char/4 fallback.
        """
        if not LLM_USAGE_METRICS:
            return
        try:
            usage = getattr(response, "usage", None)

            # Prefer API numbers (cheap attribute access).
            prompt_tokens = int(getattr(usage, "prompt_tokens", 0) or 0) if usage else 0
            completion_tokens = (
                int(getattr(usage, "completion_tokens", 0) or 0) if usage else 0
            )

            # Super-cheap fallback (~4 chars per token) if API usage is missing.
            if prompt_tokens == 0:

                def _cheap_token_estimate(s: str) -> int:
                    return (len(s) + 3) // 4 if s else 0

                # Only count string contents; ignore tool-call dicts, images, etc.
                prompt_tokens = 0
                for m in messages:
                    c = m.get("content") if isinstance(m, dict) else None
                    if isinstance(c, str):
                        prompt_tokens += _cheap_token_estimate(c)

            if completion_tokens == 0:
                ctext = ""
                try:
                    # If the model returned only tool calls, content may be empty.
                    ctext = getattr(response.choices[0].message, "content", "") or ""
                except Exception:
                    pass
                if ctext:
                    completion_tokens = (len(ctext) + 3) // 4

            # O(1) in-memory accumulation.
            self._llm_usage.prompt += int(prompt_tokens)
            self._llm_usage.completion += int(completion_tokens)
            self._llm_usage.calls += 1

        except Exception:
            # Never break the sim for usage accounting.
            pass

    def emit_llm_usage_totals(self) -> None:
        """
        Emit a single cumulative usage record for this actor to 'LLM.usage'.
        """
        if not LLM_USAGE_METRICS:
            return
        try:
            u = self._llm_usage
            total = u.prompt + u.completion
            _LLM_USAGE_LOGGER.info(
                'ACTOR totals: {"actor_id":%d,"role":"%s","calls":%d,'
                '"prompt_tokens":%d,"completion_tokens":%d,"total_tokens":%d}',
                int(self.actor_id),
                self.org_role,
                int(u.calls),
                int(u.prompt),
                int(u.completion),
                int(total),
            )
        except Exception:
            pass

    # ------------------------------------------------------------------

    def reason(self):
        """
        1) Build the prompt and log it
        2) Call the LLM with tools (tools-first)
        3) Execute any tool_calls returned (structured)
        4) Feed tool outputs back to the model, get final reply
        5) If no tool_calls, record LLM text as reasoning
        """
        self.state = "Reasoning"

        self.env.event().succeed(
            {
                "type": "Reasoning",
                "Details": {
                    "actor_id": self.actor_id,
                    "actor_type": self.org_role,
                    "time": self.env.now,
                },
            }
        )

        # 1) build prompt
        messages = self._build_reasoning_messages()

        # 2) tools-first call
        tools = self.get_openai_tools()

        prompt = self.tokenizer.apply_chat_template(
            messages,
            tools=tools,
            tool_choice="auto",
            tokenize=False,
        )

        logger.debug(role_color[self.org_role] + " prompt: %s", prompt)
        logger.debug(Fore.RESET + "-----")

        response = get_llm_chat(
            client=self.llm_client,
            messages=messages,
            tools=tools,
            tool_choice="auto",
            engine_name=self.llm_kwargs.get("engine_name") if self.llm_kwargs else None,
        )

        self._log_llm_usage(messages, response, context="reason.round1")

        # assistant's first reply (may include tool_calls)
        assistant_msg = response.choices[0].message
        self.info(f"reason(): response={assistant_msg}")
        calls = list(getattr(assistant_msg, "tool_calls", []) or [])
        self.info(
            f"reason(): tool_calls={[(c.function.name, c.id) for c in (calls or [])]}"
        )

        if calls:
            # ─────────────────────────────────────────────────────────────
            # de-dupe by tool name and allow multiple calls per turn
            # ─────────────────────────────────────────────────────────────
            MAX_TOOL_CALLS_PER_TURN = 1  # small cap to prevent spam
            seen_names = set()
            deduped = []
            for c in calls:
                n = getattr(c.function, "name", None)
                if not n:
                    continue
                if n in seen_names:
                    continue
                seen_names.add(n)
                deduped.append(c)

            calls_to_run = deduped[:MAX_TOOL_CALLS_PER_TURN]

            if len(calls) != len(calls_to_run):  # informative log if we cap/trim
                kept = ", ".join(f"{c.function.name} ({c.id})" for c in calls_to_run)
                self.logger.info(
                    f"[tool-limit] Saw {len(calls)} tool calls; executing {len(calls_to_run)} unique: {kept}"
                )

            # === run selected tools and collect results ===
            tool_results: list[
                tuple[str, str, str]
            ] = []  # (tool_call_id, tool_name, result_str)

            for c in calls_to_run:  # run each (not only the first)
                fn_name = c.function.name
                fn_args = c.function.arguments
                if isinstance(fn_args, str):
                    try:
                        fn_args = json5.loads(fn_args)
                    except Exception:
                        fn_args = {}
                # Hold the actor attention for the tool itself (existing behavior)
                call = ToolCall(
                    name=fn_name, args=fn_args or {}, sender_id=self.actor_id
                )
                call.completed_event = simpy.Event(self.env)  # to signal completion
                result_str = yield self.env.process(
                    self._run_tool_call(call)
                )  # logs CALL/RESULT
                tool_results.append((c.id, fn_name, result_str))

            # === SECOND ROUND: feed tool outputs back to the model ===
            # Re-send the assistant message (with the tool_calls we actually ran)
            second_round_LLM_messages = list(messages)

            # 1) append the assistant message **with the executed tool_calls**
            second_round_LLM_messages.append(
                {
                    "role": "assistant",
                    "content": getattr(assistant_msg, "content", None),
                    "tool_calls": [
                        {
                            "id": c.id,
                            "type": "function",
                            "function": {
                                "name": c.function.name,
                                "arguments": c.function.arguments,  # keep as-is (string or dict)
                            },
                        }
                        for c in calls_to_run  # include all we executed
                    ],
                }
            )

            # 2) append one tool message per executed call, tied by tool_call_id
            for tool_call_id, _fn_name, result_str in tool_results:  # all results
                second_round_LLM_messages.append(
                    {
                        "role": "tool",
                        "tool_call_id": tool_call_id,
                        "content": result_str or "",
                    }
                )

            # 3) ask the actor to produce a final reply
            #    Keep your "if 'ERROR' in result_str" behavior, but now consider ANY tool error.
            any_error = any(("ERROR" in (r or "")) for _, _, r in tool_results)  #

            if any_error:  # CHANGED: preserve your error-path semantics
                final_resp = get_llm_chat(
                    client=self.llm_client,
                    messages=second_round_LLM_messages,
                    tools=tools,
                    tool_choice="none",  # do not chain more tools in round 2
                    engine_name=self.llm_kwargs.get("engine_name")
                    if self.llm_kwargs
                    else None,
                )
                final_msg = final_resp.choices[0].message
                if getattr(final_msg, "content", None):
                    self.receive_message(
                        Message(
                            env=self.env,
                            content=final_msg.content,
                            sender=self.actor_id,
                            recipient=self.actor_id,
                            comm_type="reasoning",
                        )
                    )
                self._log_llm_usage(
                    second_round_LLM_messages, final_resp, context="reason.round2"
                )
                return  # done

            # ─────────────────────────────────────────────────────────────
            # also close the loop when there was NO error
            # (prevents re-requesting the same tools next step)
            # ─────────────────────────────────────────────────────────────
            final_resp = get_llm_chat(
                client=self.llm_client,
                messages=second_round_LLM_messages,
                tools=tools,
                tool_choice="none",  # stop additional tool calls in this round
                engine_name=self.llm_kwargs.get("engine_name")
                if self.llm_kwargs
                else None,
            )
            self._log_llm_usage(
                second_round_LLM_messages, final_resp, context="reason.round2"
            )
            final_msg = final_resp.choices[0].message
            if getattr(final_msg, "content", None):
                self.receive_message(
                    Message(
                        env=self.env,
                        content=final_msg.content,
                        sender=self.actor_id,
                        recipient=self.actor_id,
                        comm_type="reasoning",
                    )
                )
            return  # finished the turn after running the tools

        # No tool calls → DO NOT do a second round. Just record any text
        text = (assistant_msg.content or "").strip() if assistant_msg else ""
        if text and text != "NO_TOOL":
            self.receive_message(
                Message(
                    env=self.env,
                    content=text,
                    sender=self.actor_id,
                    recipient=self.actor_id,
                    comm_type="reasoning",
                )
            )

    def _chat_with_llm(self, messages: List[Dict[str, Any]]) -> str:
        # Enforce no tool calls in this chat
        resp = get_llm_chat(
            client=self.llm_client,
            messages=messages,
            tools=None,
            tool_choice="none",
            engine_name=self.llm_kwargs.get("engine_name"),
        )
        self._log_llm_usage(messages, resp, context="chat")
        msg = resp.choices[0].message
        return (msg.content or "").strip()

    def communicate_async(
        self, recipients: List["Actor"] = []
    ) -> Optional[str]:  # TODO: prompt LLM with message receiver
        """
        Ask the LLM for a single async **content** string.
        The orchestrator will decide who actually gets it.
        """
        self.state = "CommunicatingAsync"

        self.env.event().succeed(
            {
                "type": "CommunicatingAsync",
                "Details": {
                    "actor_id": self.actor_id,
                    "actor_type": self.org_role,
                    "time": self.env.now,
                    "recipients": [str(x.actor_id) for x in recipients],
                },
            }
        )

        # Build a simpler prompt: no JSON, just raw text reply
        messages = self._build_reasoning_messages()

        if recipients != []:
            messages[-1]["content"] += (
                f"\n\nYou are now drafting an email to the following recipients: {[x.org_role + ':' + str(x.actor_id) for x in recipients]}. Compose this short email. You can not attach files to the email. Sign with your role. Reply with ONLY the email text."
            )
        else:
            messages[-1]["content"] += (
                f"\n\nYou are now drafting an email to one of these recipients: {[a.org_role + ':' + str(a.actor_id) for a in self.simulation.actors if a is not self]}. Compose ONE short email to the most relevant recipient. You can not attach files to the email. Sign with your role. Reply with ONLY the email text."
            )

        prompt = self.tokenizer.apply_chat_template(messages, tokenize=False)

        logger.debug(
            f"\033[1;32m=== communicate_async(): prompt === \n {prompt} \n===\033[0m"
        )

        text = self._chat_with_llm(messages)

        logger.debug(
            f"\033[1;32m=== communicate_async(): LLM reply === \n {text} \n===\033[0m"
        )

        if not text:
            self.logger.info("communicate_async: empty LLM reply")
            return None
        return text

    def _ask_agenda(self, participants: List["Actor"]) -> Optional[str]:
        """
        Query the actor for a meeting agenda based on the participants and context.
        """

        messages = self._build_reasoning_messages()

        messages[-1]["content"] += (
            f"\n\nYou are now starting a meeting with the following participants: {[p.org_role + ':' + str(p.actor_id) for p in participants]}. Based on your tasks, provide a brief agenda for this meeting (one or two bullet points) for the participants. "
        )

        return self._chat_with_llm(messages)

    def communicate_sync(
        self,
        participants: List["Actor"],
        agenda: Optional[str] = None,
    ) -> "Meeting":
        """
        Initiate a Meeting, querying LLM for agenda if needed.
        """
        self.state = "CommunicatingSync"

        if agenda is None:
            # Build a terse view of active tasks for this actor
            agenda = participants[0]._ask_agenda(participants)
            self.logger.info(f"Create meeting with agenda: {agenda}")

        meeting = Meeting(self.env, participants, agenda)
        return meeting

    def _build_reasoning_messages(self) -> List[Dict[str, str]]:
        """
        Build the messages for the LLM reasoning round.
        """

        current_phase = getattr(self.simulation, "phase", None)

        task_lines = []
        for t in self.tasks.values():
            if (
                current_phase
                and current_phase not in t.phase
                and t.status != TaskStatus.COMPLETED
            ):
                continue

            line = f"- **{t.name}** ({t.description}"
            task_lines.append(line)

        tasks_block = "\n".join(task_lines)

        messages = [
            {"role": "system", "content": self.system_message},
            # {"role": "system", "content": tools_first_block},
            {
                "role": "user",
                "content": f"Context (recent):\n{self.memory.retrieve_context()}\n\n{'Tasks:' if tasks_block != '' else ''}\n{tasks_block}",
            },
        ]

        return messages

    def _run_tool_call(self, call: ToolCall):
        """
        Dispatch a single ToolCall to whichever method this actor actually
        implements, while reserving the actor's attention during the call.
        Sends the result (or error) back via receive_message.
        """
        # 0) log the tool invocation
        self.receive_message(
            Message(
                env=self.env,
                sender=self.actor_id,
                recipient=self.actor_id,
                content=f"CALL {call.name}({call.args})",
                comm_type="tool_call",
            )
        )
        self.env.event().succeed(
            {
                "type": "Tool call",
                "Details": {
                    "actor_id": self.actor_id,
                    "tool": call.name,
                    "args": call.args,
                },
            }
        )

        # 1) is the tool allowed on this actor?
        allowed = get_actor_tools(self)
        if call.name not in allowed:
            err = f"ERROR: {self.__class__.__name__} not permitted to call tool '{call.name}'"
            self.receive_message(
                Message(
                    env=self.env,
                    sender=TOOL_ACTOR_ID,
                    recipient=self.actor_id,
                    content=err,
                    comm_type="tool_call_error",
                )
            )
            return err  # return so the model sees it in the second round

        # 2) Try to acquire attention immediately (non-blocking).
        #    If attention is busy (meeting/other tool), refuse now.
        req = self.attention.request()
        if not getattr(req, "triggered", False):
            # could not acquire immediately: cancel and report busy
            try:
                req.cancel()
            except Exception:
                pass
            busy_desc = (
                f"{self.activity[0]} (remaining {self.activity[1]}h)"
                if isinstance(self.activity, tuple) and len(self.activity) == 2
                else (self.activity or "busy")
            )
            err = f"ERROR: {self.org_role} {self.actor_id} is busy ({busy_desc}). Try again later."
            self.receive_message(
                Message(
                    env=self.env,
                    sender=TOOL_ACTOR_ID,
                    recipient=self.actor_id,
                    content=err,
                    comm_type="tool_call_error",
                )
            )
            # Optional monitoring hook for busy case
            self.env.event().succeed(
                {
                    "type": "Tool busy",
                    "Details": {
                        "actor_id": self.actor_id,
                        "tool": call.name,
                        "activity": busy_desc,
                    },
                }
            )
            return err

        # 3) We have attention ― run the tool and always release at the end
        try:
            # Update task status to started if applicable
            if call.name in self.tasks:
                self.tasks[call.name].mark_started(self.actor_id)
                self.info(
                    f"Task '{self.tasks[call.name].name}' state: {self.tasks[call.name].status} by actor {self.actor_id}"
                )

            tool_fn = allowed[call.name][0]

            self._hold_attention_until_event(
                call.completed_event, activity_name=f"{call.name}"
            )

            def run_call():
                try:
                    res = tool_fn(**(call.args or {}))
                    if inspect.isgenerator(res):
                        res = yield self.env.process(res)  # type: ignore
                except Exception as e:
                    res = f"ERROR running {call.name}: {e!r}"
                call.completed_event.succeed()
                yield self.env.timeout(0)  # Make this a generator function
                return res

            try:
                result = yield self.env.process(run_call())
                # normalize to string
                if result is None:
                    result_str = ""
                elif isinstance(result, str):
                    result_str = result
                else:
                    result_str = json5.dumps(result, default=str)
            except Exception as e:
                result_str = f"ERROR running {call.name}: {e!r}"

            # Send outcome back
            self.receive_message(
                Message(
                    env=self.env,
                    sender=TOOL_ACTOR_ID,
                    recipient=self.actor_id,
                    content=f"TOOL_RESULT[{call.name}]: {result_str}",
                    comm_type="tool_result",
                )
            )
            self.env.event().succeed(
                {
                    "type": "Tool result",
                    "Details": {"actor_id": self.actor_id, "tool": call.name},
                }
            )
            # mark task as completed if applicable
            if call.name in self.tasks:
                self.tasks[call.name].mark_completed(self.actor_id)
                self.info(
                    f"Task '{self.tasks[call.name].name}' state: {self.tasks[call.name].status} by actor {self.actor_id}"
                )
            return result_str

        finally:
            # Release attention for the short tool call
            try:
                self.attention.release(req)
            except Exception:
                pass

    def participate_meeting(self, participants) -> Optional[str]:
        """
        Simulate the actor participating in a meeting.
        """

        messages = self._build_reasoning_messages()

        messages[-1]["content"] += (
            f"\n\n You are in a meeting with the following participants: {[p.org_role + ':' + str(p.actor_id) for p in participants]}. Provide your contribution to the meeting by sharing relevant information or ask questions. Output only your contribution or 'pass!' if you don't have anything meaningful to add to the conversation."
        )

        prompt = self.tokenizer.apply_chat_template(
            messages,
            tokenize=False,
        )

        logger.debug(
            f"\033[1;32m=== participate_meeting(): prompt === \n {prompt} \n===\033[0m"
        )

        response = self._chat_with_llm(messages)

        if "pass!" in response.lower():
            return None

        return response

    @property
    @abstractmethod
    def org_role(self) -> str:
        """
        Return the org_role of the actor.
        This method should be implemented by subclasses.
        """
        raise NotImplementedError("Subclasses must implement this method.")

    @property
    @abstractmethod
    def system_message(self) -> str:
        """
        Return the system prompt for the actor.
        This method should be implemented by subclasses.
        """
        raise NotImplementedError("Subclasses must implement this method.")
