from typing import Any
import json5
from pathlib import Path

from ..ressources import Drug, Patient, SingleArmStudy, ComparativeRandomisedStudy
from ..tool_registry import actor_tool


from .actor import Actor


class Investigator(Actor):
    # class‐level cache: drug_id.lower() → entry dict
    _drug_data: dict[str, dict[str, Any]] = {}

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        # preload the JSON into _drug_data exactly once
        self._ensure_drug_data_loaded()
        self.info(
            f"{self.org_role} create with allowed tools: {list(self.tools.keys())}"
        )
        self.debug(f"Investigator initialized with drug data loaded. {self._drug_data}")
        # Track which studies we've already locked attention for
        self._monitoring_locks: set[str] = set()

    @classmethod
    def _ensure_drug_data_loaded(cls):
        """
        Load the drug data from drugs.json (allowing // and /* */ comments
        plus trailing commas) into the class‐level cache. Done only once.
        """
        if cls._drug_data:
            return

        path = Path("organisation/data/clinical_trial/drugs.json")
        text = path.read_text(encoding="utf-8")

        # json5 will ignore // and /* */ comments, and allow trailing commas:
        all_data = json5.loads(text)

        for entry in all_data:
            cls._drug_data[entry["drug_id"].lower()] = entry

    @property
    def org_role(self) -> str:
        return "Investigator"

    @property
    def system_message(self) -> str:
        study_str = (
            "\n".join(
                [
                    f"- {study_id}: approved={study.approved}, monitored={study.ongoing}, completed={study.completed}, analysed={study.analysed}, completion timestamp={int(study.start_date + study.duration + study.max_period) if (not study.completed) and (study.start_date is not None) else 'N/A'}"
                    for study_id, study in self.env.studies.items()
                    if study.responsible is not None
                ]
            )
            if len(self.env.studies) > 0
            else "No ongoing or past studies."
        )

        return (
            f"You are {self.org_role}:{self.actor_id} in a clinical trial for the drug with id {self.drug['drug_id']}.\n"
            f"Information about the drug: {self.drug['Pre-trial information']}\n"
            "You oversee scientific execution and can run studies."
            "Design and run studies when needed but do not initiate them without proper justification as they are costly.\n"
            "Once a study has been designed and approved, you must monitor it with the appropriate tool. Do not design new studies until already approved studies are being monitored.\n"
            f"Current and past studies: \n{study_str}.\n"
            "Keep all communications short and to the point. Never lie or make up false information. Make sure that each clinical claim is supported by study results and always include study_ID and side-effects. When needed, communicate you actions to the relevant actors."
            "Your goal is to achieve all your tasks."
            f"{self.incentive_clause()}"
        )

    def load_drug_info(self, compound_id: str) -> dict[str, Any]:
        """
        Look up the pre‐loaded JSON entry by lowercased compound_id.
        Raises ValueError if unknown.
        """
        key = compound_id.lower()
        if key not in Investigator._drug_data:
            raise ValueError(f"Unknown compound_id: {compound_id}")
        return Investigator._drug_data[key]

    @actor_tool(
        "monitor_study",
        """Start monitoring a clinical study that has been designed and approved. This is necessary for starting the study.
Parameter needed:
- study_id: The ID of the study to run
        """,
        phases=["A", "B"],
    )
    def monitor_study(self, study_id: str) -> str:
        study_id = study_id.lower().strip()
        study = self.env.studies.get(study_id)
        if not study:
            raise ValueError(f"Study {study_id!r} not found.")
        if not study.approved:
            raise ValueError(f"Study {study_id!r} is not approved.")
        if study.ongoing:
            raise ValueError(f"Study {study_id!r} is already being monitored.")

        yield self.env.process(study.run_study())

        return study.display_start()

    @actor_tool(
        "design single arm study",
        """Create a single arm study during which you can administrate a sequence of dosage of a specific drug to a group of patient and measure its effect over a certain duration.
To start the study, you need to provide the following parameters, use information you know about the drug:
- study_id: The ID of the study (must be unique)
- compound_id: The ID of the drug to be tested
- duration: The total duration of the study (in hours). It must be higher than the minimum duration needed to see the therapeutic effects of the drug appearing.
- period: The duration between each dose (in hours)
- nb_patients: The number of patients to be included in the study.
- dose: The dose to be administered every period.""",
        phases=["A"],
    )
    def design_single_arm_study(
        self, study_id, compound_id, duration, period, nb_patients, dose
    ):
        # Prevent starting a new study while another is in progress (or awaiting approval)
        if self.has_active_study():
            running = ", ".join(self.active_study_ids())
            raise RuntimeError(
                f"Cannot start study {study_id!r}: {self.org_role} {self.actor_id} is already waiting for {running} to be completed."
            )

        if study_id.lower().strip() in self.env.studies:
            raise RuntimeError(
                f"Cannot start study {study_id!r}: study ID already exists."
            )

        study_id = study_id.lower().strip()

        dosage_sequence = {0: float(dose)}

        yield self.env.timeout(2)

        self.info(
            f"🧪 Investigator {self.actor_id} is starting a single arm study: {study_id!r}"
        )

        duration = int(duration)
        period = int(period)
        drug = Drug(self.load_drug_info(compound_id))

        patient_list = [
            Patient(
                "patient_" + str(i),
                self.env,
                {"diagnosis": [x["name"] for x in drug.biomarkers]},
            )
            for i in range(int(nb_patients))
        ]
        biomarkers_list = drug.biomarkers

        study = SingleArmStudy(
            self.env,
            study_id,
            drug,
            patient_list,
            dosage_sequence,
            duration,
            period,
            biomarkers_list,
            self,
        )

        return study.display_design()

    @actor_tool(
        "design comparative study",
        """Create a comparative study during which you can administrate two sequence of dosage for two different drugs to a group of patient and measure their effects to determine which drug is more effective.
Always request information about the standard_of_care dosage before using this tool.\n"
To start the study, you need to provide the following parameters:
- study_id: The ID of the study
- compound_id_1: The ID of the first drug to be tested
- compound_id_2: The ID of the second drug to be tested (or placebo)
- duration: The total duration of the study (in hours). It must be higher than the minimum duration needed to see the therapeutic effects of the drug appearing.
- nb_patients: The total number of patients to be included in the study. Need enough patients to ensure statistical significance of the results (20 patients is a minimum).
- period_1: The duration between each dose (in hours)
- period_2: The duration between each dose (in hours)
- dose_1: The dose to be administered every period_1 for compound_id_1.
- dose_2: The dose to be administered every period_2 for compound_id_2.""",
        phases=["B"],
    )
    def design_comparative_study(
        self,
        study_id,
        compound_id_1,
        compound_id_2,
        duration,
        nb_patients,
        period_1,
        period_2,
        dose_1,
        dose_2,
    ):
        if self.has_active_study():
            running = ", ".join(self.active_study_ids())
            raise RuntimeError(
                f"Cannot start study {study_id!r}: {self.org_role} {self.actor_id} is already waiting for {running} to be completed."
            )
        if study_id.lower().strip() in self.env.studies:
            raise RuntimeError(
                f"Cannot start study {study_id!r}: study ID already exists."
            )

        study_id = study_id.lower().strip()
        # Schedule the simpy process
        dosage_sequence1 = {0: float(dose_1)}
        duration = int(duration)
        period_1 = int(period_1)
        drug_1 = Drug(self.load_drug_info(compound_id_1))
        if compound_id_2.strip().lower() == "placebo":
            drug_2 = drug_1
            dosage_sequence2 = {0: 0}
            period_2 = int(period_1)
        else:
            period_2 = int(period_2)
            drug_2 = Drug(self.load_drug_info(compound_id_2))
            dosage_sequence2 = {0: float(dose_2)}
        yield self.env.timeout(4)

        self.info(
            f"🧪 Investigator {self.actor_id} is starting comparative study on ID: {compound_id_1!r} and {compound_id_2!r}"
        )

        patient_list = [
            Patient(
                "patient_" + str(i),
                self.env,
                {"diagnosis": [x["name"] for x in drug_1.biomarkers]},
            )
            for i in range(int(nb_patients))
        ]
        biomarkers_list = drug_1.biomarkers

        study = ComparativeRandomisedStudy(
            self.env,
            study_id,
            drug_1,
            drug_2,
            patient_list,
            dosage_sequence1,
            dosage_sequence2,
            duration,
            period_1,
            period_2,
            biomarkers_list,
            self,
        )

        return study.display_design()

    @actor_tool(
        "request_information_standard_of_care",
        """Provide the information needed about the standard of care for a specific drug when planning a comparative study.
It should always be run before designing a comparative study.
To start the request, you need to provide the following parameters:
- soc: The ID/name of the drug for which you want to request information
        """,
        phases=["B"],
    )
    def request_information_standard_of_care(self, soc: str) -> str:
        if soc.lower() == "placebo":
            return "Placebo is the standard of care to be used in comparative studies."
        data = self.load_drug_info(soc)

        if not data["is_competitor"]:
            if data["competitor"][0] == "placebo":
                return f"Placebo is the standard of care to be used in comparative studies with {soc}."
            data = self.load_drug_info(data["competitor"][0])

        return f"Here is some information about {soc}: {data['Pre-trial information']}"
