from ..messages import Message
from ..tool_registry import actor_tool

from .actor import Actor


class MedicalExpert(Actor):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.info(
            f"{self.org_role} create with allowed tools: {list(self.tools.keys())}"
        )

    @property
    def org_role(self) -> str:
        return "Medical Expert"

    @property
    def system_message(self) -> str:
        return (
            "You are a Medical Expert in a clinical trial. "
            "Your role is to provide medical expertise and advice. "
            "You have access to the tool `evaluate_patient_data`, which analyzes patient data "
            "and provides insights on treatment efficacy and safety. "
            "When you run evaluate_patient_data, **you MUST** invoke the tool "
            "`TOOL[evaluate_patient_data](...)` rather than writing a free‐form answer."
            "Always respond with either a tool invocation or, if no tool is needed, a short justification."
            f"{self.incentive_clause()}"
        )

    @actor_tool(
        "evaluate_patient_data",
        "Evaluate patient data for treatment efficacy and safety. Args: {'patient_id': str}",
        phases=["A"],
    )
    def evaluate_patient_data(self, patient_id: str) -> str:
        """
        Tool that evaluates patient data.
        """
        # Log evaluation
        self.info(f"🔍 Evaluating patient data for ID: {patient_id!r}")

        # Notify all actors of evaluation
        for actor in self.simulation.actors:
            msg = Message(
                env=self.env,
                content=f"🔍 Patient data evaluated for ID: {patient_id}",
                sender=f"{self.actor_id}:MedicalExpert",
                recipient=f"{actor.actor_id}:{actor.__class__.__name__}",
                comm_type="tool_result",
            )
            actor.receive_message(msg)

        return f"Patient data for ID {patient_id} has been evaluated."
