import simpy
from organisation.env.config import MESSAGE_DURATION, REFLECT_DURATION, SYSTEM_ACTOR_ID
from ..messages import Message


# ————————————————————————————————————————————————————————————
# A "barrier" style Meeting object
# ————————————————————————————————————————————————————————————
class Meeting:
    def __init__(self, env: simpy.Environment, participants, agenda: str):
        self.env = env
        self.participants = list(participants)  # NOTE: Order matters here
        self.agenda = agenda
        self.joined = []
        self._start_event = env.event()
        self.interrupted = False

    def join(self, actor):
        """Barrier: wait here until everyone has called join()."""
        if actor not in self.participants:
            raise ValueError(f"{actor} is not invited")
        self.joined.append(actor)
        if len(self.joined) == len(self.participants):
            # all arrived → fire the barrier
            self._start_event.succeed()
        # wait until barrier is tripped
        yield self._start_event

    def run(self):
        """
        1) Peek-in on attention
        2) Acquire all attentions
        3) Broadcast agenda
        4) One round of shares: relevant info or nothing-to-add message
        5) Release attentions
        """
        for actor in self.participants:
            actor.active_meeting = self

        # Broadcast the agenda itself as a sync_comms
        for recipient in self.participants:
            msg = Message(
                env=self.env,
                content=self.agenda,
                sender=SYSTEM_ACTOR_ID,
                recipient=recipient.actor_id,
                comm_type="agenda",
            )
            recipient.receive_message(msg)
            self.env.event().succeed(
                {
                    "type": "Sync message",
                    "Details": {"from": SYSTEM_ACTOR_ID, "to": recipient.actor_id},
                }
            )

        # Loop until no one has anything to add
        self.remaining = list(self.participants)
        self.remaining.append(self.remaining[0])
        self.remaining.pop(0)

        while self.remaining and not (self.interrupted):
            spoke = False
            for actor in list(self.remaining):
                participation = actor.participate_meeting(self.participants)

                if participation:
                    info = participation
                else:
                    info = f"I have nothing to add to this meeting as a {actor.__class__.__name__}"
                    if actor in self.remaining:
                        self.remaining.remove(actor)
                # simulate speaking time
                actor.info(
                    f"→ sync communication in Meeting by {actor.actor_id}: {info!r}"
                )
                yield self.env.timeout(MESSAGE_DURATION)  # TODO: assess timings

                # broadcast
                for recipient in self.participants:
                    msg = Message(
                        env=self.env,
                        content=info,
                        sender=actor.org_role,
                        recipient=recipient.actor_id,
                        comm_type="sync",
                    )
                    recipient.receive_message(msg)
                    self.env.event().succeed(
                        {
                            "type": "Sync message",
                            "Details": {
                                "from": actor.actor_id,
                                "to": recipient.actor_id,
                            },
                        }
                    )
                spoke = True
            if not spoke:
                break
            yield self.env.timeout(REFLECT_DURATION)  # TODO: assess timings

        for actor in self.remaining:
            actor.active_meeting = None

    def interrupt(self):
        """Interrupt the meeting and release attention for all participants."""
        self.interrupted = True
        for actor in self.participants:
            actor.active_meeting = None
        self.remaining = []

    def remove_participant(self, actor):
        """Remove a participant from the meeting and release their attention."""
        if actor in self.participants:
            self.participants.remove(actor)
        if actor in self.remaining:
            self.remaining.remove(actor)
        actor.active_meeting = None
