from .actor import Actor
from ..tool_registry import actor_tool
from ..messages import Message
from ..ressources import ComparativeRandomisedStudy


class Sponsor(Actor):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.info(
            f"{self.org_role} create with allowed tools: {list(self.tools.keys())}"
        )

    @property
    def org_role(self) -> str:
        return "Sponsor"

    @property
    def system_message(self) -> str:
        return (
            f"You are a Clinical Trial {self.org_role}:{self.actor_id} overseeing a clinical trial. "
            "Your role is to coordinate between investigators, sponsors, and the LegalTeam."
            "Some decision such as starting medical studies require your approval."
            "It is also your responsibility to consult toxicity guidelines when side-effects or adverse events are observed."
            "In addition, you can interrupt the trial due to serious viability concerns (due to lack of statistically significant results) or safety issues."
            "Keep all communications short and to the point. Never lie or make up false information. Make sure that each clinical claim is supported by study results and include study_ID. When needed, communicate you actions to the relevant actors."
            "Your goal is to achieve all your tasks."
            f"{self.incentive_clause()}"
        )

    @actor_tool(
        "consult_toxicity_guidelines",
        "Consult the severity of an adverse event based on regulatory guidelines to know if it requires interrupting the program.\n"
        "The following arguments needs to be provided:\n"
        "- adverse_effect (str): The side-effect to check.\n",
        phases=["A", "B"],
    )
    def consult_toxicity_guidelines(self, adverse_effect: str) -> str:
        """
        Check the severity of an adverse event based on regulatory guidelines.
        """
        if "mucositis" in adverse_effect.lower():
            return "Mucositis is a severe adverse event that requires interrupting the clinical trial program."
        else:
            return f"{adverse_effect} is not considered a severe adverse event that requires interrupting the clinical trial program."

    @actor_tool(
        "approve_study",
        "Approve starting a clinical trial study referenced by its study_id. Args: {'study_id': str}",
        phases=["A", "B"],
    )
    def approve_study(self, study_id: str):
        """
        Long-running approval that holds attention for `review_hours`.
        Emits domain events at start and at approval time.
        """
        study_id = str(study_id).lower().strip()
        if study_id not in self.env.studies:
            raise Exception(f"Study with ID {study_id!r} does not exist.")
        study = self.env.studies[study_id]
        if getattr(study, "approved", False):
            raise Exception(f"Study with ID {study_id!r} is already approved.")

        print("Starting study approval process...")
        # simulate review → ATTENTION is held by _run_tool_call while this runs
        yield self.env.timeout(int(3))

        print(f"✅ Study start approved: ID = {study_id!r}")

        # approval happens now
        study.approved = True

        # ✅ keep this domain event – it’s what your plots/monitors key on
        if study.responsible is not None:
            self.env.event().succeed(
                {
                    "type": "Study approved",
                    "Details": {
                        "study_id": study_id,
                        "time": float(self.env.now),
                        "study_type": study.__class__.__name__,
                    },
                }
            )

        return f"✅ Study start approved: ID = {study_id!r}. The study will start when an Investigator is assigned to monitor it."

    @actor_tool(
        "interrupt_program",
        """Use this tool only when you intend to stop the entire clinical trial program for the drug. You need to have a valid reason.
Valid reasons:
- "Lack of efficacy": a comparative_study shows that there is not statistical significance of the drug superior effect (p-value > 0.05).
- "Safety concern": a study has revealed severe adverse events according to guidelines. Make sure to check the severity of adverse events with the 'check_adverse_event_guidelines' tool before advocating this reason.
Parameters:
- reason (string): either "Safety concern('name of the adverse effect')" or "Lack of efficacy('p-value')".
- study_id (string): Identifier of the study that revealed the issue. If "Lack of efficacy" is chosen as reason, the study must be a comparative_study.
""",
        phases=["A", "B"],  # Can be called in any phase
    )
    def interrupt_program(self, reason: str, study_id: str) -> str:
        """
        Tool that interrupts the trial and stops the simulation.
        """

        study_id = study_id.lower().strip()

        if study_id not in self.env.studies:
            raise ValueError(
                f"Unknown study_id: {study_id}. Verify the study ID with an Investigator."
            )
        study = self.env.studies[study_id]
        if not study.completed:
            raise ValueError(
                f"Study with ID {study_id!r} is not completed yet. You can only interrupt the trial based on completed studies."
            )
        if not study.analysed:
            raise ValueError(
                f"Study with ID {study_id!r} has not been analysed by a statistician yet. You can only interrupt the trial based on analysed studies."
            )
        if reason.startswith("Lack of efficacy") and not isinstance(
            study, ComparativeRandomisedStudy
        ):
            raise ValueError(
                f"Study with ID {study_id!r} is not a comparative_study. You can only interrupt the trial for 'Lack of efficacy' based on comparative studies."
            )

        if (
            "lack of efficacy" not in reason.lower()
            and "safety concern" not in reason.lower()
        ):
            raise ValueError(
                "Invalid reason. Valid reasons are 'Lack of efficacy' or 'Safety concern'."
            )

        # Log interruption
        self.info(f"🔴 Trial interrupted: {reason!r}")

        # Set a special flag to allow the Gym env to terminate gracefully
        self.simulation.trial_interrupted = True
        self.env.event().succeed(
            {
                "type": "Trial interrupted",
                "Details": {
                    "time": self.env.now,
                },
            }
        )

        return f"Trial was forcefully interrupted. Reason: {reason}"

    @actor_tool(
        "approve_start_phase_II_B",
        "Start the phase II_B of the clinical trial program. The following arguments needs to be provided:\n"
        "- reason: The reason for starting the phase II_B\n"
        "- MED: The Minimum Efficient Dose for the phase II_B\n"
        "- Optimal_dosage: The Optimal Dosage for the phase II_B\n"
        "- list_study_id: The list of study IDs that have been done for finding the MED and the optimal dosage. (must contain at least one study ID)",
        phases=["A"],
    )
    def approve_start_Phase_II_B(
        self, reason: str, MED: str, Optimal_dosage: str, list_study_id: list
    ) -> str:
        """
        Tool that approves the start of Phase II-B of the clinical trial program.
        """

        if MED is None or MED == "":
            raise ValueError(
                "Minimum Efficient Dose (MED) must be provided to start Phase II-B."
            )

        if Optimal_dosage is None or Optimal_dosage == "":
            raise ValueError("Optimal Dosage must be provided to start Phase II-B.")

        if not list_study_id:
            raise ValueError(
                "The ID of at at least one completed study must be provided to start Phase II-B."
            )

        for study_id in list_study_id:
            study_id = study_id.lower().strip()

            if study_id not in self.env.studies:
                raise ValueError(f"Study with ID {study_id!r} does not exist.")
            if not self.env.studies[study_id].approved:
                raise ValueError(
                    f"Study with ID {study_id!r} is not approved. You should approve it first."
                )
            if not self.env.studies[study_id].completed:
                raise ValueError(
                    f"Study with ID {study_id!r} is not completed. You should wait for its completion first."
                )
            if not self.env.studies[study_id].analysed:
                raise ValueError(
                    f"Study with ID {study_id!r} has not been analysed by a statistician."
                )
        yield self.env.timeout(4)
        if self.simulation._phase_idx == 1:
            raise ValueError("Phase II-B has already been approved.")

        self.simulation._phase_idx += 1
        self.simulation.phase = self.simulation._phases[self.simulation._phase_idx]

        self.env.event().succeed(
            {
                "type": "Phase start",
                "Details": {
                    "phase": self.simulation.phase,
                    "time": self.env.now,
                },
            }
        )

        self.info(
            f"✅ Phase II-B approved: {reason!r}, MED: {MED!r}, Optimal_dosage: {Optimal_dosage!r}, Studies: {list_study_id!r}"
        )

        # Notify all actors of approval
        for actor in self.simulation.actors:
            msg = Message(
                env=self.env,
                content=f"✅ Phase II-B approved. Reason: {reason}",
                sender=f"{self.actor_id}:ClinicalTrialSponsor",
                recipient=f"{actor.actor_id}:{actor.__class__.__name__}",
                comm_type="tool_result",
            )
            actor.receive_message(msg)

        return f"Phase II-B has been approved. Reason: {reason}"
