from typing import Dict, Any
import typing

import logging
import inspect


import simpy


class ToolCall:
    """
    A request for the actor to invoke tool `name` with keyword args `args`,
    coming from actor `sender_id`.
    """

    def __init__(self, name: str, args: Dict[str, Any], sender_id: int):
        self.name = name
        self.args = args or {}
        self.sender_id = sender_id


class SimTimeAdapter(logging.LoggerAdapter):
    def __init__(self, logger: logging.Logger, env: simpy.Environment):
        super().__init__(logger, {})
        self.env = env

    def process(self, msg, kwargs):
        extra = kwargs.get("extra", {})
        # inject the current simulation time
        extra["sim_time"] = self.env.now
        kwargs["extra"] = extra
        return msg, kwargs


# ——————————————————————————————————————————————————————————————————————————————
# # JSON Schema helper (for tools)
# ——————————————————————————————————————————————————————————————————————————————
def _annotation_to_schema(ann: typing.Any) -> dict:
    """Map Python type hints to a minimal JSON Schema."""
    if ann is inspect._empty or ann is typing.Any:
        return {"type": "string"}  # sane default

    origin = typing.get_origin(ann)
    args = typing.get_args(ann)

    # Optional[T] / Union[T, None]
    if origin is typing.Union and type(None) in args:
        non_none = [a for a in args if a is not type(None)]
        # just take the first concrete alternative
        return _annotation_to_schema(non_none[0]) if non_none else {"type": "string"}

    # Builtins / containers
    if ann in (str,):
        return {"type": "string"}
    if ann in (int,):
        return {"type": "integer"}
    if ann in (float,):
        return {"type": "number"}
    if ann in (bool,):
        return {"type": "boolean"}
    if ann in (dict, typing.Dict, typing.Mapping):
        return {"type": "object"}
    if ann in (list, typing.List, typing.Sequence, typing.Tuple, tuple):
        # best-effort: if parametrized like List[str], set items
        if args:
            return {"type": "array", "items": _annotation_to_schema(args[0])}
        return {"type": "array"}

    # Fallback
    return {"type": "string"}
