import os
import json
import logging
from pathlib import Path
from typing import Any, Dict, Literal, Optional

from openai import AzureOpenAI, OpenAI

from organisation.env.config import (
    LLM_ENGINE,
    QWEN_API_BASE,
    LLM_TEMPERATURE,
    LLM_MAX_TOKENS,
    QWEN_MODEL,
    GPT5_MODEL,  # deployment name for Azure chat
)

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


# ─────────────────────────────────────────────────────────────
# tiny .env loader (no external deps)
# ─────────────────────────────────────────────────────────────
def _find_project_root() -> Path:
    here = Path(__file__).resolve()
    for p in [here, *here.parents]:
        if (p / ".env").exists():
            return p if p.is_dir() else p.parent
    return Path.cwd()


def _load_env_file(dotenv_path: Path) -> None:
    try:
        with dotenv_path.open("r", encoding="utf-8") as f:
            for line in f:
                s = line.strip()
                if not s or s.startswith("#") or "=" not in s:
                    continue
                k, v = s.split("=", 1)
                k = k.strip()
                v = v.strip().strip('"').strip("'")
                os.environ.setdefault(k, v)
    except FileNotFoundError:
        pass


# ─────────────────────────────────────────────────────────────
# clients
# ─────────────────────────────────────────────────────────────
def create_llm_client(
    engine_name: Literal["AzureOpenAI", "qwen"] = LLM_ENGINE,
) -> Any:
    """
    Minimal, explicit client factory:
      - AzureOpenAI: requires endpoint, api_version, api_key in environment (.env).
      - Qwen/vLLM: OpenAI-compatible at QWEN_API_BASE; api_key optional.
    """
    _load_env_file(_find_project_root() / ".env")
    engine_name = engine_name or LLM_ENGINE

    if engine_name == "AzureOpenAI":
        endpoint = os.getenv("AZURE_OPENAI_ENDPOINT")
        api_version = os.getenv("AZURE_OPENAI_API_VERSION")
        api_key = os.getenv("AZURE_OPENAI_API_KEY")
        missing = [
            k
            for k, v in {
                "AZURE_OPENAI_ENDPOINT": endpoint,
                "AZURE_OPENAI_API_VERSION": api_version,
                "AZURE_OPENAI_API_KEY": api_key,
            }.items()
            if not v
        ]
        if missing:
            raise RuntimeError(
                f"Missing required Azure env vars: {', '.join(missing)}. "
                "Put them in your .env (no quotes, no spaces around '=')."
            )
        return AzureOpenAI(
            azure_endpoint=endpoint, api_version=api_version, api_key=api_key
        )

    if engine_name == "qwen":
        api_key = os.getenv("VLLM_API_KEY", "EMPTY")
        return OpenAI(api_key=api_key, base_url=QWEN_API_BASE)

    raise ValueError(f"Unknown engine name: {engine_name}")


# ─────────────────────────────────────────────────────────────
# chat
# ─────────────────────────────────────────────────────────────
def get_llm_chat(
    client: Any,
    messages: list[Dict[str, Any]],
    *,
    tools: Optional[list[dict]] = None,
    tool_choice: Optional[dict | str] = "auto",
    reasoning: Optional[bool] = True,
    engine_name: Literal["AzureOpenAI", "qwen"] = LLM_ENGINE,
) -> Any:
    """
    Tools-first chat completion for both Azure and Qwen/vLLM.
    No hidden config lookups—everything comes from .env and config.py.
    """
    engine_name = engine_name or LLM_ENGINE

    if engine_name == "AzureOpenAI":
        # deployment name (a.k.a. 'model' parameter for Chat Completions in Azure)
        deployment = os.getenv("AZURE_OPENAI_DEPLOYMENT", GPT5_MODEL)
        if not deployment:
            raise RuntimeError(
                "Azure deployment name not set. Define AZURE_OPENAI_DEPLOYMENT in .env "
                "or set GPT5_MODEL in config.py."
            )

        kwargs: Dict[str, Any] = dict(
            model=deployment,
            messages=messages,
            temperature=LLM_TEMPERATURE,
            max_tokens=LLM_MAX_TOKENS,
            stream=False,
        )
        if tools:
            kwargs["tools"] = tools
        if tool_choice is not None:
            kwargs["tool_choice"] = tool_choice

        return client.chat.completions.create(**kwargs)

    if engine_name == "qwen":
        kwargs: Dict[str, Any] = dict(
            model=QWEN_MODEL,
            messages=messages,
            temperature=LLM_TEMPERATURE,
            max_tokens=LLM_MAX_TOKENS,
            # vLLM supports extra_body; we keep it minimal
            extra_body={"add_generation_prompt": True},
        )
        if tools:
            kwargs["tools"] = tools
        if tools and tool_choice is not None:
            kwargs["tool_choice"] = tool_choice
        if reasoning is False:
            kwargs["extra_body"]["chat_template_kwargs"] = {"enable_thinking": False}

        return client.chat.completions.create(**kwargs)

    raise ValueError(f"Unknown engine name: {engine_name}")


# ─────────────────────────────────────────────────────────────
# utilities
# ─────────────────────────────────────────────────────────────
def extract_tool_calls(resp: Any) -> list[dict]:
    """
    Normalize tool calls into: [{name: str, arguments: dict}]
    Works for both AzureOpenAI and OpenAI-compatible responses.
    """
    msg = resp.choices[0].message
    tool_calls = getattr(msg, "tool_calls", None)
    out: list[dict] = []

    if tool_calls:
        for c in tool_calls:
            fun = c.function
            args = fun.arguments
            if isinstance(args, str):
                try:
                    args = json.loads(args)
                except Exception:
                    pass
            out.append({"name": fun.name, "arguments": args})
    return out
