import sys
import logging
from pathlib import Path
from typing import Union


def setup_sim_logging(env, output_dir: Union[Path, str], level: int = logging.INFO):
    """
    One place to configure formatting + sim_time injection + console & file handlers.
    Call this once in run scripts after you construct `env`.
    """
    root = logging.getLogger()
    # Clear existing handlers
    for h in list(root.handlers):
        root.removeHandler(h)

    # Ensure we always have a sim_time on records (non-Actor logs)
    _orig_factory = logging.getLogRecordFactory()

    def record_factory(*args, **kwargs):
        record = _orig_factory(*args, **kwargs)
        if not record.name.startswith("Actor["):
            record.sim_time = getattr(record, "sim_time", 0.0)
        return record

    logging.setLogRecordFactory(record_factory)

    fmt = logging.Formatter(
        fmt="[sim={sim_time:.1f}] {name} {levelname}: {message}", style="{"
    )

    def inject_sim_time(record: logging.LogRecord):
        # pull current time from the env *at emission*
        record.sim_time = env.simulation.env.now
        return True

    console = logging.StreamHandler(sys.stdout)
    console.setFormatter(fmt)
    console.addFilter(inject_sim_time)

    fileh = logging.FileHandler(Path(output_dir) / "simulation.log", mode="a")
    fileh.setFormatter(fmt)
    fileh.addFilter(inject_sim_time)

    root.addHandler(console)
    root.addHandler(fileh)
    root.setLevel(level)

    # --- Dedicated events logger -> logs/<timestamp>/events.log ---
    events_logger = logging.getLogger("monitoring.events")
    events_logger.setLevel(level)
    events_logger.propagate = False  # don't duplicate into simulation.log

    events_file = logging.FileHandler(Path(output_dir) / "events.log", mode="a")
    events_file.setFormatter(fmt)
    events_file.addFilter(inject_sim_time)
    events_logger.addHandler(events_file)
