import numpy as np


def biomarker_sigmoid(D, dose_response, baseline):
    """
    Sigmoid model for biomarker response to drug dose.

    Parameters:
    - D: Drug dose
    - baseline: Initial biomarker level
    - dose_response: Dictionary containing model parameters.

    Returns:
    - Returns the new biomarker level after applying the sigmoid model.
    """
    noise_drug = np.random.normal(0, dose_response["parameters"].get("std", 0))
    biomarker_std = dose_response["biomarker"].get("biomarker_std", 0)
    noise_biomarker = np.random.normal(0, biomarker_std)

    gamma = dose_response["parameters"].get("gamma", 1)
    E0 = dose_response["parameters"].get("E0", 0)
    Emax = dose_response["parameters"].get("Emax", 1)
    EC50 = dose_response["parameters"].get("EC50", 1)

    # print("noise_drug", noise_drug, "noise_biomarker", noise_biomarker)

    if dose_response["parameters"].get("Absolute", False):
        new_level = (
            baseline
            + E0
            + ((Emax - E0) * (D**gamma) / (EC50**gamma + D**gamma))
            + noise_drug
            + noise_biomarker
        )
    else:
        new_level = (
            baseline
            * (1 + E0 + (Emax * (D**gamma) / (EC50**gamma + D**gamma)) + noise_drug)
            * (1 + noise_biomarker)
        )

    return new_level


def biomarker_exponential(D, dose_response, baseline):
    """
    Exponential model for biomarker response to drug dose.

    Parameters:
    - D: Drug dose
    - baseline: Initial biomarker level
    - dose_response: Dictionary containing model parameters.

    Returns:
    - Returns the new biomarker level after applying the exponential model.
    """
    noise_drug = np.random.normal(0, dose_response["parameters"].get("std", 0))
    biomarker_std = dose_response["biomarker"].get("biomarker_std", 0)
    noise_biomarker = np.random.normal(0, biomarker_std)

    k = dose_response["parameters"]["k"]
    Emax = dose_response["parameters"].get("Emax", 0)

    new_level = (
        baseline
        * (1 + Emax * (1 - np.exp(-k * D)) + noise_drug)
        * (1 + noise_biomarker)
    )

    return new_level


def biomarker_constant(D, dose_response, baseline):
    """
    Constant model for biomarker response to drug dose.

    Parameters:
    - D: Drug dose
    - baseline: Initial biomarker level
    - dose_response: Dictionary containing model parameters.

    Returns:
    - Returns the new biomarker level after applying the constant model.
    """
    noise_drug = np.random.normal(0, dose_response["parameters"].get("std", 0))
    biomarker_std = dose_response["biomarker"].get("biomarker_std", 0)
    noise_biomarker = np.random.normal(0, biomarker_std)

    threshold = dose_response["parameters"]["threshold"]
    Emax = dose_response["parameters"].get("Emax", 0)
    E0 = dose_response["parameters"].get("E0", 0)

    r = np.random.rand()

    if D >= threshold:
        new_level = int(r < Emax + noise_drug + noise_biomarker)
    else:
        new_level = int(r < E0 + noise_drug + noise_biomarker)

    return new_level


def risk_exponential(D, side_effect):
    """
    Exponential risk model for side effect occurrence based on drug dose.

    Parameters:
    - D: Current drug dose
    - side_effect: Dictionary containing side effect parameters.
    - old_max: Previous maximum cumulative dose.

    Returns:
    - Returns the risk of side effect occurrence.
    """
    return max(
        side_effect["min_risk"],
        min(
            side_effect["max_risk"],
            (side_effect["min_risk"] + 1e-10)
            * (
                side_effect["max_risk"] / (side_effect["min_risk"] + 1e-10)
            )  # prevent division by zero
            ** (
                (D - side_effect["min_dose"])
                / (side_effect["max_dose"] - side_effect["min_dose"])
            ),
        ),
    )
    # Discount the risk based on the lack of side effect in the past


def risk_constant(D, side_effect):
    """
    Constant risk model for side effect occurrence based on drug dose.

    Parameters:
    - D: Current drug dose
    - side_effect: Dictionary containing side effect parameters.

    Returns:
    - Returns the risk of side effect occurrence.
    """

    if D >= side_effect["min_dose"]:
        return side_effect["max_risk"]
    else:
        return side_effect["min_risk"]
