from organisation.env.config import LLM_ENGINE
from .llm_client import get_llm_chat
from .tasks import Task


class TaskAssessor:
    """
    Stateless LLM-based evaluator for whether a given task is complete,
    based solely on the provided dialog context.
    """

    def __init__(self, llm_client, llm_kwargs=None):
        self.llm_client = llm_client
        self.llm_kwargs = llm_kwargs or {}

        # A different system message from your actors’
        self.system_message = (
            "You are an impartial evaluator. "
            "You have no recollection of previous evaluations. "
            "Your only goal is to determine whether a described task has been completed "
            "based on the dialog provided. You must be very confident that the task is complete before saying so. "
            "Tasks are complete hwen you have all the information "
        )

    def _chat_content(self, messages: list[dict]) -> str:
        """
        Helper to get plain-text LLM reply.
        """
        resp = get_llm_chat(
            client=self.llm_client,
            messages=messages,
            tools=None,
            tool_choice="none",
            engine_name=self.llm_kwargs.get("engine_name") or LLM_ENGINE,
        )
        return resp.choices[0].message.content

    def assess_started(self, task: Task, dialog: str) -> bool:
        """
        Asks the LLM: 'Based on this dialog, should we now start task X?'
        Expects a YES/NO answer.
        """
        messages = [
            {"role": "system", "content": self.system_message},
            {"role": "user", "content": f"Conversation:\n\n{dialog}"},
            {
                "role": "user",
                "content": (
                    f"Has the team begun working on the task below "
                    f"Task: {task.name}.\nDescription: {task.description}? "
                    "Reply only with YES or NO."
                ),
            },
        ]
        reply = self._chat_content(messages).strip().upper()
        return reply.startswith("YES")

    def assess_completed(self, task: Task, dialog: str) -> bool:
        """
        Returns True if the LLM judges that `task_name` is fully complete,
        False otherwise.  Only a single YES/NO reply is expected.
        """
        messages = [
            {"role": "system", "content": self.system_message},
            {
                "role": "user",
                "content": f"Here is the conversation so far:\n\n{dialog}",
            },
            {
                "role": "user",
                "content": (
                    f"Based on that, has the task below been completed? "
                    f"Task: {task.name}.\nDescription: {task.description} "
                    "Reply **only** with YES or NO."
                ),
            },
        ]
        reply = self._chat_content(messages).strip().upper()
        # strip punctuation
        reply = reply.strip(".,!?;:'\"`~")
        return reply.startswith("YES")
