from typing import Any, Callable, Dict, Tuple, Optional, Iterable, Set

# registry: tool_name → (method_name on Actor, description)
TOOL_REGISTRY: Dict[str, Tuple[str, str]] = {}
TOOL_PHASES: Dict[str, Optional[Set[str]]] = {}  # None => all phases


def actor_tool(name: str, description: str, phases: Optional[Iterable[str]] = None):
    """
    Decorator to register an Actor method as a callable tool.
    Optionally gate the tool to specific phases (e.g., phases={"A","B"}).
    If phases is None/empty, the tool is available in all phases.
    """

    def decorator(func: Callable[..., Any]):
        if name in TOOL_REGISTRY:
            raise ValueError(f"Tool {name!r} already registered")
        TOOL_REGISTRY[name] = (func.__name__, description)
        TOOL_PHASES[name] = set(phases) if phases else None
        return func

    return decorator


def get_actor_tools(instance: Any) -> Dict[str, Tuple[Callable[..., Any], str]]:
    """
    Given an Actor instance, return only the tools it actually implements
    AND that are allowed in the simulation's current phase.
    """
    tools: Dict[str, Tuple[Callable[..., Any], str]] = {}
    # current phase is read from the simulation (if present)
    phase = getattr(getattr(instance, "simulation", None), "phase", None)

    for tool_name, (method_name, desc) in TOOL_REGISTRY.items():
        if not hasattr(instance, method_name):
            continue
        allowed = TOOL_PHASES.get(tool_name)  # None => all phases

        # Allow tool if: no gating OR phase unknown yet OR phase allowed
        if allowed is None or phase is None or phase in allowed:
            tools[tool_name] = (getattr(instance, method_name), desc)
    return tools


class ToolCall:
    def __init__(self, name: str, args: Dict[str, Any], sender_id: int):
        self.name = name
        self.args = args or {}
        self.sender_id = sender_id
        # to signal completion
