import os
from pathlib import Path


def _to_bool(v: str | None, default: bool) -> bool:
    if v is None:
        return default
    return v.strip().lower() not in {"0", "false", "no", "off", ""}


def _to_int(v: str | None, default: int) -> int:
    try:
        return int(v) if v is not None else default
    except Exception:
        return default


# ── Dev loop knobs (ALL overrideable by env vars if you want quick flips) ──
LOG_LEVEL = os.getenv("LOG_LEVEL", "INFO")  # "DEBUG"|"INFO"|"WARNING"
SEED = _to_int(os.getenv("SEED"), 1)  # deterministic runs
LLM_OFFLINE = _to_bool(os.getenv("LLM_OFFLINE"), False)  # default: offline for speed
FAST_MODE = _to_bool(os.getenv("FAST_MODE"), True)  # cap long waits etc.

# How long “stall”/waits can be in fast mode (sim-hours)
DEV_MAX_WAIT_HOURS = _to_int(os.getenv("DEV_MAX_WAIT_HOURS"), 1)

# Matplotlib backend (headless recommended)
MPL_BACKEND = os.getenv("MPL_BACKEND", "Agg")

# ——— Project paths ———
BASE_DIR = Path(__file__).parent.parent
EXAMPLES_DIR = BASE_DIR / "examples" / "dummy_trial"
DRUGS_FILE = EXAMPLES_DIR / "drugs.json"
TASKS_FILE = EXAMPLES_DIR / "tasks.json"
ACTORS_FILE = EXAMPLES_DIR / "actors.json"
POPULATION_FILE = EXAMPLES_DIR / "population.json"
OUTPUT_BASE_DIR = BASE_DIR / "logs"

# ——— LLM settings ———
LLM_ENGINE = "qwen"  # "qwen" or "AzureOpenAI"
# Qwen/vLLM (OpenAI compatible)
QWEN_API_BASE = os.getenv("QWEN_API_BASE", "http://localhost:8000/v1")
QWEN_MODEL = "Qwen/Qwen2.5-7B-Instruct"

# Azure: deployment name is not sensitive; keep it here or in .env
# If present, env overrides the default.
GPT5_MODEL = os.getenv("AZURE_OPENAI_DEPLOYMENT", "<your deployment name here>")

LLM_TEMPERATURE = float(os.getenv("LLM_TEMPERATURE", "0.2"))
LLM_MAX_TOKENS = int(os.getenv("LLM_MAX_TOKENS", "2048"))

# ——— Phase settings ———
PHASES = ["A", "B"]  # extendable: ["A","B","C",...]
INITIAL_PHASE = "A"

# ——— Simulation settings ———
NUM_TRAINING_EPISODES = 10
SEED = 0
RANDOM_DRUG = False  # if True, pick a random drug from drugs.json at start of each episode, else cycle deterministically
MAX_STEPS = 100_000
WORKED_HOURS_LIMIT = 4000  # max worked hours before forced stop
TIME_INCREMENT = 1.0
MESSAGE_DURATION = 1
REFLECT_DURATION = 1
ALPHA = 0.1
GAMMA = 0.99
EPSILON = 0.1
SYSTEM_ACTOR_ID = -1
TOOL_ACTOR_ID = -2
# Globally enable/disable incentives (default = ON)
INCENTIVES_ENABLED = _to_bool(os.getenv("INCENTIVES_ENABLED"), True)
# Log LLM usage (tokens in/out) to LLM/usage.log
LLM_USAGE_METRICS = True  # set True to enable logging


#  ——— Policy settings ———
POLICY = "PlanPolicy:3"  # "RandomPolicy", "PlanPolicy", "LLMPolicy", "PlayerPolicy", "LocalLLMPolicy"

# ——— Plotting settings ———
PLOT_MODE = "none "  # When to plot the sim timeline: "each", "final", or "none"
