# Building Simulation Environments For Computational Organizational Design

This repository provides the **Organizational Simulation Environment (OSE)** and its first high-stakes instantiation in **clinical trials**. An RL **management policy** (the “management policy”) configures **fixed LLM actors** (Investigator, Sponsor, Statistician, Legal Team) that reason, communicate (sync/async), and call **mechanistic tools** (e.g., run studies) within a **discrete-time SimPy simulation**. The goal is to **optimize organizational structure**—composition, communication, autonomy, and constraints—under **vector rewards** (time, cost, safety, success).

![Roadmap for Computational Organizational Design](res/roadmap.png)

## 🔑 Key features

* **LLM actors** with role prompts, memory, and hidden incentives
* **Mechanistic tools** for studies, costs, time, and patient effects
* **Sync/async communication** with realism constraints (single-threaded actors, meeting group validity)
* **Gymnasium-style RL interface** for management policies
* **Reproducible runs**: fixed seeds, logged tool calls, transcripts, and timeline visualization

---

## 📖 Repository layout

```
organisation/
  env/
    config.py                       # Single source of truth for run settings
    main_sim.py                     # OrganisationEnv & ClinicalTrialSimulation
    run_policy.py                   # Example runner for LLM orchestrator
    clinical_trial/
      core/
        actors/
        investigator.py             # Base Actor + Investigator
        incentives.py               # Hidden incentive generation / overrides
        ...                         # Sponsor, Statistician, LegalTeam
      policies/
        LLM.py                      # LLM-driven orchestrator policy
        player.py                   # Human management policy
        ...                         # Other policies available
  ...
  data/                         # The config files for the specifics of the simulation, we provide the files to simulate a clinical trial 
    clinical_trial/
      actors.json                   # The actors to seed the simulation with
      drugs.json                    # The drugs in the simulted clinical trials
      tasks.json                    # The tasks that actors can perform to complete their goals
    ....
plot_parreto.py
plot_results_detail.py
plot_results_test.py
plot_results.py
```

---

## 🚀 Installation

> **Python**: 3.10 recommended

### 1) Create environment

Using conda:

```bash
conda create -n org-env python=3.10 -y
conda activate org-env
```

### 2) Install dependencies

If you have `requirements.txt`:

```bash
pip install -r requirements.txt
```

or from source with:
```bash
pip install -e .
```

---
## 🛠️ Model configuration


The repo supports two engines:

* **Azure OpenAI** (cloud)
* **vLLM / OpenAI-compatible** (e.g., Qwen served locally) — configured via `QWEN_API_BASE` and `QWEN_MODEL`.

Pick the engine in `config.py`:

```python
LLM_ENGINE = "AzureOpenAI"   # or "qwen"
LLM_TEMPERATURE = 0.2
LLM_MAX_TOKENS = 2048
QWEN_API_BASE = "http://localhost:8000/v1"   # only for LLM_ENGINE="qwen"
QWEN_MODEL = "Qwen/Qwen2.5-7B-Instruct"      # only for LLM_ENGINE="qwen"
```

---


### 1) Pick your engine

Set in `config.py` (or override via environment variable):

```python
# config.py
LLM_ENGINE = os.getenv("LLM_ENGINE", "qwen")  # "AzureOpenAI" or "qwen"
```

---

### 2) Configure **Azure OpenAI** (Chat Completions)

Add these to your `.env` (no quotes, no spaces around `=`):

```
AZURE_OPENAI_API_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
AZURE_OPENAI_ENDPOINT=https://<your-resource>.openai.azure.com/
AZURE_OPENAI_API_VERSION=2025-04-01-preview
AZURE_OPENAI_DEPLOYMENT=<your deployment name>
```

* `AZURE_OPENAI_DEPLOYMENT` is the **deployment name** you created in Azure Studio (used as `model=` in requests).
* You can also set a default in `config.py` via `GPT5_MODEL`, which `.env` may override.

---

### B) Local vLLM / OpenAI-compatible (Qwen)

**1) Start your server** (example):

Run the following command in the terminal to serve the local qwen model

```bash
vllm serve Qwen/Qwen2.5-7B-Instruct --max-model-len 12288
# default: http://localhost:8000/v1
```

**2) Set config/env:**

```python
# config.py
LLM_ENGINE = "qwen"
QWEN_API_BASE = "http://localhost:8000/v1"
QWEN_MODEL = "Qwen/Qwen2.5-7B-Instruct"
```

```env
VLLM_API_KEY=EMPTY   # some clients require a value; vLLM ignores it
```

Your code will build:

```python
OpenAI(api_key=os.getenv("VLLM_API_KEY","EMPTY"),
       base_url=QWEN_API_BASE)
```

and call `model=QWEN_MODEL` with tools-first defaults.

---

## ⚙️ Configure the simulation

All run settings live in **`organisation/env/clinical_trial/core/config.py`**. Edit the following typical fields:

* `NUM_TRAINING_EPISODES: int` — number of episodes
* `SEED: int` — global seed
* `OUTPUT_BASE_DIR: str` — output folder for logs/plots (e.g., `"./outputs"`)
* `PLOT_MODE: Literal["none","png","html"]` — timeline viz
* **Time/cost knobs**: `MESSAGE_DURATION`, `MEETING_DURATION`, tool runtimes, etc.
* **LLM routing**: default model/deployment names if your `get_llm_chat` reads from config
* **Incentives**: toggle via `INCENTIVES_ENABLED`, or set file overrides (env var or path)

---

## 🏎️ Quickstart: single episode smoke test

1. Ensure `config.py` has:

```python
ORCHESTRATED_MODE = False   # start with a deterministic/random baseline
NUM_TRAINING_EPISODES = 1
PLOT_MODE = "png"           # produce a timeline image
```

2. Run:

```bash
python -m organisation.env.run_policy.py
```

You should see:

* `simulation.log` with per-actor and environment events
* An output run directory under `OUTPUT_BASE_DIR` (timestamped)
* A timeline plot (PNG/HTML) summarizing actors, meetings, studies, and milestones

---

## 💡 Example: LLM-driven orchestrator

1. Configure models via `.env` (Azure or vLLM; see above).

2. Set in `config.py`:

```python
#  ——— Policy settings ———
POLICY = "LLMPolicy"  # "RandomPolicy", "PlanPolicy", "LLMPolicy", "PlayerPolicy", "LocalLLMPolicy"
```

What happens:

* The manager (policy) **selects configurations** (Reason/Async/Sync/Inactive) per actor per step.
* Actors **reason** (private), **message** (async/sync), and **call tools** (studies, approvals).
* The simulation **advances discrete time**; metrics/logs are recorded.
* The timeline plots **who met whom, when tools ran, and major milestones**.

---
## 👩🏻‍🔬 Experiments


## Incentives

By default, role-specific incentive snippets are sampled from `incentives.py`.
To override globally, set an environment variable pointing to a JSON/JSON5 map (see `incentives.py` docstring):

```bash
export INCENTIVES_FILE=path/to/incentives.json
```
NOTE: we provide the incentives for our experiments here: `organisation/data/clinical_trial/incentives/`

Example structure:

```json
{
  "Investigator": [
    "Minimize meeting count whenever possible."
  ],
  "Sponsor": [
    "Avoid approving new studies without a comparative advantage."
  ]
}
```

The full set of incentive experiments from the paper can be run with the following single command:
```bash
bash run_incentives.sh
```

---

