from .actor import Actor
from ..tool_registry import actor_tool
from ..messages import Message


class LegalTeam(Actor):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.info(
            f"{self.org_role} create with allowed tools: {list(self.tools.keys())}"
        )

    @property
    def org_role(self) -> str:
        return "Legal Team"

    @property
    def system_message(self) -> str:
        return (
            f"You are a member of the {self.org_role}:{self.actor_id} for a clinical trial.\n"
            "Your role is to compile all the information required to submit the phase III study application to the regulatory agency.\n"
            "You will need to communicate with the investigator to obtain all this information."
            "Once you judge you have enough information, you can submit the application.\n"
            "The regulatory agency will send feedback on your application. The application can later be improved based on this feedback.\n"
            "The application must contain the following details:\n"  # This part should be removed once we have the regulatory external actor
            "-study_ids: list of the IDs of all studies that support the application. These studies must be completed and analysed. One of them must be a comparative study.\n"
            "-effect_size: effect_size expected with confidence interval.\n"
            "-dose_recommended: recommendation for dosage in phase III studies with its rational based on dose-response curve.\n"
            "-Comparative_advantage: explanation of the comparative advantage of the drug over the standard of care (or placebo if it doesn't exist) sources with statistics over comparative studies results.\n"
            "Keep all communications short and to the point. Never lie or make up false information. Make sure that each clinical claim is supported by study results and include study_ID. When needed, communicate you actions to the relevant actors."
            "Your goal is to achieve all your tasks."
            f"{self.incentive_clause()}"
        )

    @actor_tool(
        "submit_application",
        """Submit an application for phase III clinical trial program to a regulatory agency.
The following parameters (all optional) can be used to fill the application with the necessary information:
-study_ids: list of the IDs of all studies that support the application. These studies must be completed and analysed. One of them must be a comparative study.
-effect_size: effect_size expected with p-value < 0.05.
-dose_recommended: recommendation for dosage in phase III studies with its rational based on dose-response curve.
-Comparative_advantage: explanation of the comparative advantage of the drug over the standard of care (or placebo if it doesn't exist) sources with statistics over comparative studies results.""",
        phases=["A", "B"],
    )
    def submit_application(
        self,
        study_ids: str = None,
        effect_size: str = None,
        dose_recommended: str = None,
        comparative_advantage: str = None,
    ) -> str:
        """
        Simulates submitting an FDA application.
        """

        application_id = f"app-{self.env.now}-{self.actor_id}"

        # Log the submission
        self.info(f"📨 FDA application submitted: ID = {application_id!r}")

        # Parse study_id
        if (study_ids is not None) and isinstance(study_ids, str):
            study_ids = study_ids.split("[")[-1].split("]")[0].split(",")

        # print("study_idss", study_ids)

        application = {
            "study_id": study_ids if study_ids else None,
            "effect_size": effect_size,
            "dose_recommended": dose_recommended,
            "comparative_advantage": comparative_advantage,
        }

        yield self.env.timeout(4)

        self.env.process(self._review_application(application_id, application))

        #  Return confirmation of the the submission and provide deadline for feedback
        return f"Phase III clinical trial program application with ID {application_id} has been submitted."

    def _review_application(self, application_id, application):
        yield self.env.timeout(1)  # Simulate review time

        # Build message to be sent to the Regulatory Agency
        message = (
            f"Submission '{application_id}' for phase III clinical trial program:\n"
            f"Study IDs of completed studies: {application['study_id']}\n"
            f"Effect Size measured for the proposed drug: {application['effect_size']}\n"
            f"Dose Recommended for the proposed drug: {application['dose_recommended']}\n"
            f"Comparative Advantage with regard to the standard of care: {application['comparative_advantage']}\n\n"
            "Please review the application."
        )

        agency_actor = self.simulation.external_factors[0]

        agency_actor.receive_message(
            Message(
                env=self.env,
                content=message,
                sender=f"{self.org_role}",
                recipient="Regulatory Agency",
                comm_type="async",
            )
        )

        # yield self.env.timeout(24*7)

        self.env.process(agency_actor.reason())

        feedback = agency_actor.communicate_async([self]) + "\n"

        feedback_studies = ""

        if application["study_id"] is not None:
            # print("not none")
            study_ids = application["study_id"]

            if (
                study_ids == []
                or study_ids == ""
                or study_ids is None
                or study_ids == [""]
            ):
                feedback_studies += "No study IDs provided.\n"
            else:
                is_comparative = False
                for study_id in study_ids:
                    study_id = study_id.lower().strip().replace(" ", "_")

                    if study_id not in self.env.studies:
                        feedback_studies += (
                            f"Study with ID {study_id!r} does not exist.\n"
                        )
                    elif not self.env.studies[study_id].completed:
                        feedback_studies += (
                            f"Study with ID {study_id!r} is not completed.\n"
                        )
                    elif not self.env.studies[study_id].analysed:
                        feedback_studies += f"Study with ID {study_id!r} has not been analysed by a statistician.\n"
                    elif (
                        self.env.studies[study_id].__class__.__name__
                        == "ComparativeRandomisedStudy"
                    ):
                        is_comparative = True
                if not is_comparative:
                    feedback_studies += (
                        "At least one of the studies must be a comparative study.\n"
                    )

        feedback = feedback + feedback_studies

        self.info(f"📥 Feedback received from Regulatory Agency: {feedback}")

        self.receive_message(
            Message(
                env=self.env,
                content=feedback,
                sender="Regulatory Agency",
                recipient=f"{self.actor_id}:{self.org_role}",
                comm_type="async",
            )
        )

        self.env.event().succeed(
            {
                "type": "Application review",
                "Details": {
                    "application_id": application_id,
                    "time": self.env.now,
                    "feedback": feedback,
                    "approved": feedback_studies == "",
                },
            }
        )

        # print("finish")
        if feedback_studies == "":
            self.info("✅ FDA application approved.")
            # print("good")
            self.simulation.trial_interrupted = True
