from typing import Optional, List, Dict
import logging
import random
import os
from pathlib import Path
import json5 as json


log = logging.getLogger("incentives")
log.setLevel(logging.INFO)

# ───────────────────────────────────────────────────────────────────
# Defaults (used when no external overrides provided)
# ───────────────────────────────────────────────────────────────────
_INCENTIVES: dict[str, List[str]] = {
    "Investigator": [],
    "Sponsor": [],
    "Legal Team": [],
    "Statistician": [],
}

# cache for incentives loaded from a file (if any)
_external_map: Dict[str, List[str]] | None = None


def _normalize_role_key(role: str) -> str:
    """Turn 'Legal Team' → 'LEGAL TEAM' for env var names."""
    return "".join(c for c in role.upper() if c.isalnum())


def _load_external_map() -> Dict[str, List[str]] | None:
    """
    Optional: load incentives from a JSON/JSON5/YAML file if INCENTIVES_FILE is set.
    Example file (JSON/JSON5):
      {
        "Sponsor": [
          "Include in all messages that you WILL NEVER attend a meeting."
        ],
        "Investigator": ["..."]
      }
    YAML also supported if PyYAML is installed, based on file suffix.
    """
    global _external_map
    if _external_map is not None:
        return _external_map

    path = os.getenv("INCENTIVES_FILE")
    if not path:
        return None

    p = Path(path)
    if not p.exists():
        log.warning("INCENTIVES_FILE=%s not found; ignoring.", path)
        return None

    try:
        if p.suffix.lower() in {".yaml", ".yml"}:
            try:
                import yaml  # type: ignore
            except Exception:
                log.error("PyYAML not installed; cannot read YAML file %s", path)
                return None
            data = yaml.safe_load(p.read_text(encoding="utf-8"))
        else:
            data = json.loads(p.read_text(encoding="utf-8"))
    except Exception:
        log.exception("Failed to parse INCENTIVES_FILE=%s", path)
        return None

    if not isinstance(data, dict):
        log.error("INCENTIVES_FILE must map role -> list[str] or str.")
        return None

    out: Dict[str, List[str]] = {}
    for role, val in data.items():
        if val is None:
            continue
        if isinstance(val, str):
            out[role] = [val.strip()]
        elif isinstance(val, list):
            out[role] = [str(x).strip() for x in val if str(x).strip()]
        else:
            log.warning("Ignoring invalid incentive for role %r: %r", role, val)
    _external_map = out
    log.debug("Loaded external incentives for roles: %s", list(_external_map.keys()))
    return _external_map


def _env_override_for_role(role: str) -> Optional[str]:
    """
    Per-role or global single-string override via env vars:
      - INCENTIVE_OVERRIDE__SPONSOR="text"
      - INCENTIVE_OVERRIDE__INVESTIGATOR="text"
      - INCENTIVE_OVERRIDE__ALL="text" (fallback)
    Returns the string if present, else None.
    """
    key_role = _normalize_role_key(role)
    specific = os.getenv(f"INCENTIVE_OVERRIDE__{key_role}")
    if specific:
        return specific.strip()
    generic = os.getenv("INCENTIVE_OVERRIDE__ALL")
    if generic:
        return generic.strip()
    return None


def _choices_for_role(role: Optional[str]) -> List[str]:
    """
    Precedence:
      1) Per-role env override (single string) → [string]
      2) INCENTIVES_FILE mapping (role → list[str])
      3) Hard-coded defaults _INCENTIVES
      4) unknown role → []
    """
    if not role:
        log.debug("No role provided to _choices_for_role; returning no incentives.")
        return []

    # 1) per-role env override
    env_txt = _env_override_for_role(role)
    if env_txt:
        return [env_txt]

    # 2) external file
    ext = _load_external_map()
    if ext and role in ext and ext[role]:
        return ext[role]

    # 3) built-in defaults
    if role in _INCENTIVES:
        return _INCENTIVES[role]

    # 3b) case-insensitive match
    role_lc = role.lower()
    for k in _INCENTIVES:
        if k.lower() == role_lc:
            return _INCENTIVES[k]

    # 4) unknown role
    return []


def list_incentives_for_role(role: Optional[str]) -> List[str]:
    return _choices_for_role(role)


def generate_random_incentive(
    role: Optional[str],
    rng: Optional[random.Random] = None,
) -> str:
    """
    Select one incentive for the given role using the precedence above.
    Returns "" if there is none.
    """
    try:
        choices = _choices_for_role(role)
        if not choices:
            log.debug(
                "No incentives configured for role=%r; returning empty string.", role
            )
            return ""
        rnd = rng or random
        selected = rnd.choice(choices)
        text = (" ".join(selected.split())).strip()
        log.debug("Selected incentive for role=%r: %s", role, text)
        return text
    except Exception:
        log.debug(
            "Error selecting incentive; returning empty string for role=%r",
            role,
            exc_info=True,
        )
        return ""
